/*
  int32Math.h - 32bit math that uses 32bit instructions on 32bit CPUs.
  Copyright 2012-2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef __int32Math_h
#define __int32Math_h

/** @file
 * @brief Math functions using 32bit integer instructions only.
 */

#include <stdbool.h>
#include <integers.h>

inline static Int32 int32Abs(Int32 v) {
	return v>=0 ? v : -v;
}

inline static Int32 int32Max(Int32 a, Int32 b) {
	return a>b ? a : b;
}

inline static Int32 int32Min(Int32 a, Int32 b) {
	return a<b ? a : b;
}

/** Mathematical exact DIV operation, that rounds down to -infinity even for negative values.
 * The remainder is always non-negative.
 * @param a the dividend.
 * @param b the divisor. If this is negative, then the sign of dividend and divisor are flipped.
 * @return the quotient of a/b rounded towards -infinity.
 */
Int32 int32MathDiv(Int32 a, Int32 b);

/** Mathematical exact DIV operation, that rounds down to -infinity even for negative values.
 * @param a the dividend.
 * @param b the divisor. If this is negative, then the sign of dividend and divisor are flipped.
 * @return the remainder of the integer division. This is always in the range 0..abs(b)-1 .
 */
Int32 int32MathMod(Int32 a, Int32 b);

/** Maps a value to the closest 'more discrete' value.
 * @param step the increment of the discrete steps. 0 is the first value, 0+step the second and so on.
 * @param value the value to quantize. May be positive or negative.
 * @return that value n*step that is closest to value.
 */
Int32 int32Quantize(Int32 step, Int32 value);

/** Similar to int32Quantize, except that the step value must be a power of 2.
 * @param log2Step the increment of the discrete steps. 0 is the first value, 0+(1<<logStep) the second and so on.
 * @param value the value to quantize. May be positive or negative.
 * @return that value n*(1<<logStep) that is closest to value.
 */
Int32 int32QuantizeExp2(Int32 log2Step, Int32 value);

Uint32 uint32Power(Uint32 x, Uint32 y);

/** Reverses the order of the lower bits of an integer.
 * @param value the bit pattern to reverse
 * @param length the number of bits to reverse.
 * @return a pattern where bits 0..length-1 are mapped to bits length-1..0.
 */
Uint32 uint32BitReverse(Uint32 value, Uint32 length);

/** Rounds correctly at +0.5 even if the values are negative.
 * @param value the value representation to round. The real value will be value/divider.
 * @param divider the divider used to get the real value.
 * @return a rounded representation, that after dividing will yield the correctly rounded result.
 */
inline static Int32 int32RoundForDivide(Int32 value, Int32 divider) {
	return value + (value>=0 ? divider>>1 : -divider>>1);
}

/** Linear interpolation between two points. Only 32-bit calculations are used for
 * improved performance which limits the values. The formula used is
 *
 * r = ( (x1-x)*y0 + (x-x0)*y1 + (x1-x0)/2 ) / (x1-x0)
 *
 * None of the values might be out of the range -2^31..+2^31-1.
 * Interpolation results lie between the two extremes, no extrapolation is performed.
 */
Int32 int32LinearInterpolateFast(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x);

/** Linear interpolation between two points.
 * Interpolation results lie between the two extremes if x lies between the extremes. Extrapolation
 * is performed. Care must be taken not to overflow the integer ranges, calculations are performed
 * using 32-bit integers only but products of (xi-xj) and ys are summed up. Keep (x1-x0) small and x
 * between these values or keep ys small.
 */
Int32 int32LinearExtrapolateFast(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x);

/** Integer square root, rounded towards 0.
 * @return the largest integer r such, that r*r <= x
 */
Int32 uint32SqrtFloor(Uint32 x);

/** Calculates the integral part log2, rounded up.
 * @param x a value >=1, 0 is allowed as an exception.
 * @return the smallest value l such that 2^l >= x. This value is in the range of 0..32 (yes, 32).
 * If x==0, then -32 is returned.
 */
inline static int uint32Log2Ceil(Uint32 x) {
	if (x==0) return -32;
	for (int i=0; i<32; ++i) if (1<<i >= x) return i;
	return 32;
}

/** Calculates the integral part log2, rounded down.
 * @param x a value >= 1.
 * @return the biggest value l such that 2^l <= x . This value is in the range of 0..31.
 * If x==0 then -32 is returned (good approximation of 0 for Int32).
 */
inline static int uint32Log2Floor(Uint32 x) {
	for (int l=31; l>=0; --l) if (1<<l <= x) return l;
	return -32;	// approximation of 0
}

/** Shift function allowing positive and negative counts.
 * @param a the value to shift
 * @param p the number of bits to shift, positive or negative.
 * @return the signed shift result.
 */
static inline Int32 int32MulExp2(Int32 a, int p) {
	if (p>=0) return a << p;
	else return a >> -p;
}

/** Shift function allowing positive and negative counts.
 * @param a the value to shift
 * @param p the number of bits to shift, positive or negative.
 * @return the signed shift result.
 */
static inline Uint32 uint32MulExp2(Uint32 a, int p) {
	if (p>=0) return a << p;
	else return a >> -p;
}

#endif
