/*
  int64Math.h - math functions, that tend to be slow on 32bit CPUs.
  Copyright 2012-2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef int64Math_h
#define int64Math_h

#include <integers.h>

/** @file
 * @brief 64-bit integer math.
 *
 * This module provides functions, that may be considerably less efficient than the corresponding int32-functions
 * on a 32-bit processor because of emulation of 64-bit calculations.
 */


inline static Int64 int64Abs(Int64 v) {
	return v>=0 ? v : -v;
}

inline static Int64 int64Max(Int64 a, Int64 b) {
	return a>b ? a : b;
}

inline static Int64 int64Min(Int64 a, Int64 b) {
	return a<b ? a : b;
}

/** Unsigned divide 64-bit integer (software).
 */
Uint64 uint64Div(Uint64 a, Uint64 b);

Uint64 uint64Mod(Uint64 a, Uint64 b);

#ifdef NATIVE_64BIT
/** Signed divide 64-bit integer.
 */
static inline Int64 int64Div(Int64 a, Int64 b) {	return a/b;	}
static inline Int64 int64Mod(Int64 a, Int64 b) {	return a%b;	}

#else
Int64 int64Div(Int64 a, Int64 b);
Int64 int64Mod(Int64 a, Int64 b);
#endif

/** Calculates the integral square root approximation.
 * @param r radicant
 * @return the biggest value x such that x*x <= r
 */
int uint64SqrtFloor(Uint32 r);

/** Calculates the integral square root approximation.
 * @param r radicant
 * @return the smallest positive value x such that x*x >= r
 */
int uint64SqrtCeil(Uint32 r);

/** Calculates the integral part log2, rounded up.
 * @param x a value >=1, 0 is allowed as an exception.
 * @return the smallest value l such that 2^l >= x. This value is in the range of 0..64 (yes, 64).
 * If x==0, then -64 is returned.
 */
inline static int uint64Log2Ceil(Uint64 x) {
	if (x==0) return -64;
	for (int l=0; l<64; ++l) if (1llu<<l >= x) return l;
	return 64;
}

/** Calculates the integral part log2, rounded down.
 * @param x a value >= 1.
 * @return the biggest value l such that 2^l <= x . This value is in the range of 0..63.
 * If x==0 then -64 is returned (good approximation of 0 for Int64).
 */
inline static int uint64Log2Floor(Uint64 x) {
	for (int l=63; l>=0; --l) if (1<<l <= x) return l;
	return -64;	// approximation of 0
}

/** Shift function allowing positive and negative counts.
 * @param a the value to shift
 * @param p the number of bits to shift, positive or negative.
 * @return the signed shift result.
 */
static inline Int64 int64MulExp2(Int64 a, int p) {
	if (p>=0) return a << p;
	else return a >> -p;
}

/** Shift function allowing positive and negative counts.
 * @param a the value to shift
 * @param p the number of bits to shift, positive or negative.
 * @return the signed shift result.
 */
static inline Uint64 uint64MulExp2(Uint64 a, int p) {
	if (p>=0) return a << p;
	else return a >> -p;
}

/** Rounds correctly at +0.5 even if the values are negative.
 * @param value the value representation to round. The real value will be value/divider.
 * @param divider the divider used to get the real value.
 * @return a rounded representation, that after dividing will yield the correctly rounded result.
 */
static inline Int64 int64RoundForDivide(Int64 value, Int64 divider) {
	return value + (value>=0 ? divider>>1 : -divider>>1);
}

/** Performs linear interpolation of 32-bit values. Because of huge intermediate values, this calculation requires
 * 64-bit math. That's why it's found here. There are no limitations on the values. This function does NOT perform
 * extrapolation outside the range x0 to x1. y1 and y0 are the result extremes.
 * @param x0 x-value of first defining point.
 * @param y0 y-value of first defining point.
 * @param x1 x-value of second defining point.
 * @param y1 y-value of second defining point.
 * @param x the position of the interpolation request.
 * @return the linearly interpolated value between (including) y0 and y1.
 */
Int32 int32LinearInterpolate(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x);

/** Performs linear interpolation of 32-bit values. Because of huge intermediate values, this calculation requires
 * 64-bit math. That's why it's found here. Rhis function performs
 * extrapolation outside the range x0 to x1. It is the user's responsibility to avoid range overflows. There will not
 * be any overflows in the range x0 to x1.
 * @param x0 x-value of first defining point.
 * @param y0 y-value of first defining point.
 * @param x1 x-value of second defining point.
 * @param y1 y-value of second defining point.
 * @param x the position of the interpolation request.
 * @return the linearly interpolated value.
 */
Int32 int32LinearExtrapolate(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x);

#endif

