//HEADER
/*
  lpcMemories-sliced.c 
  Copyright 2011 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#include <lpcMemories.h>
#include <macros.h>

//SLICE
const LpcFamilyMember* lpcFindById(const LpcFamilyMember *const *list, Uint32 id) {
	for (;*list; list++) if ((*list)->id == id) return *list;
	return 0;
}

//SLICE
const LpcFamilyMember* lpcFindByNumber(const LpcFamilyMember *const *list, int number) {
	for (;*list; list++) if ((*list)->number == number) return *list;
	return 0;
}

//SLICE
/** Calculates what sectors are affected from operations within an address range.
 * @param sectorFrom the first sector affected
 * @param sectorTo the last sector affected
 * @param addressFrom the lowest address of the range
 * @param addressTo the highest address of the range.
 * @return true, if the addresses are within the members FLASH range
 */
bool lpcAddressRangeToSectorRange(LpcFamilyMember const *member,
	int *sectorFrom, int *sectorTo, Uint32 addressFrom, Uint32 addressTo) {

	Uint32 from,to;
	int sector;
	for (sector = 0; lpcSectorRange(member,&from,&to,sector); sector++) {
		if (from<=addressFrom && addressFrom<=to) {
			*sectorFrom = sector;
			break;
		}
	}
	for ( ; lpcSectorRange(member,&from,&to,sector); sector++) {
		if (from<=addressTo && addressTo<=to) {
			*sectorTo = sector;
			return true;
		}
	}
	return false;
}

//SLICE
int lpcAddressToSector(LpcFamilyMember const *member, Uint32 address) {
	int sectorFrom, sectorTo;
	if (lpcAddressRangeToSectorRange(member,&sectorFrom,&sectorTo,address,address)) return sectorFrom;
	else return -1;
}

//SLICE
bool lpcSectorRange(LpcFamilyMember const *member, Uint32 *addressFrom, Uint32 *addressTo, int sector) {
	const LpcFamilyMemoryMap *map = member->familyMemoryMap;

	Uint32 offsetFlash = 0;
	int subSector = 0;
	int sectorType = 0;
	for (int s=0; sectorType<ELEMENTS(map->sectorArrays) && map->sectorArrays[sectorType].sizeK!=0 ; s++) {
		if (s==sector) {
			*addressFrom = map->addressFlash + offsetFlash;
			*addressTo = *addressFrom + 1024*map->sectorArrays[sectorType].sizeK -1;
			return true;
		}
		offsetFlash += 1024*map->sectorArrays[sectorType].sizeK;
 
		// next one
		if (subSector+1 < map->sectorArrays[sectorType].n) subSector ++;
		else {
			sectorType++;
			subSector = 0;
		}
	}
	return false;	// outside FLASH bounds of device
}

//SLICE
int lpcFamilySectorHighest(LpcFamilyMemoryMap const *mm) {
	int sector = 0;
	for (int ss=0; ss<ELEMENTS(mm->sectorArrays); ++ss) {
		sector += mm->sectorArrays[ss].n;
	}
	return sector -1;
}

//SLICE
int lpcFamilySectorOffset(LpcFamilyMemoryMap const *mm, int sector) {
	int offset = 0;
	int sectorCounter = 0;
	for (int ss=0; ss<ELEMENTS(mm->sectorArrays); ++ss) {
		for (int i=0; i<mm->sectorArrays[ss].n; ++i,++sectorCounter) {
			if (sector==sectorCounter) return offset;
			offset += mm->sectorArrays[ss].sizeK * 1024;
		}
	}
	return sectorCounter==sector ? offset : -1;
}

//SLICE
Uint32 lpcCrpLevelToCode(int level) {
	switch(level) {
		case 1:	return 0x12345678u;
		case 2: return 0x87654321u;
		case 3: return 0x43218765u;
		default: return 0xFFFFFFFFu;
	}
}

//SLICE
int lpcCrpCodeToLevel(Uint32 code) {
	switch(code) {
		case 0x12345678u: return 1;
		case 0x87654321u: return 2;
		case 0x43218765u: return 3;
		default: return 0;
	}
}

//SLICE
bool fifoPrintLpcFamilyMember(Fifo *f, LpcFamilyMember const *m) {
	const LpcFamilyMemoryMap* mm = m->familyMemoryMap; 
	bool success =
		fifoPrintString(f,"LPC") && fifoPrintUDec(f,m->number,4,10)
		&& fifoPrintString(f,", ID=0x") && fifoPrintHex(f,m->id,8,8)
		&& fifoPrintString(f,"\nFLASH: ") && fifoPrintUDec(f,m->sizeFlashK,1,10)
		&& fifoPrintString(f,"kiB @ 0x") && fifoPrintHex(f,mm->addressFlash,8,8)
		&& fifoPrintString(f,"\nRAM:   ")
		;
	for (int r=0; r<ELEMENTS(m->sizeRamKs) && success; r++) if ( m->sizeRamKs[r]!=0) {
		if (r!=0) success = success && fifoPrintString(f,", ");
		success = success
		&& fifoPrintUDec(f,m->sizeRamKs[r],1,10)
		&& fifoPrintString(f,"kiB @ 0x")
		&& fifoPrintHex(f,mm->addressRams[r],8,8)
		;
	}

	success = success && fifoPrintLn(f);
	Uint32 offset = 0;
	int sector = 0;
	for (int ss=0; offset<m->sizeFlashK*1024 && ss<ELEMENTS(mm->sectorArrays); ++ss) {
		for (int s=0; s<mm->sectorArrays[ss].n && offset<m->sizeFlashK*1024; ++s) {
			success	= success
			&& fifoPrintString(f,"Sector ")
			&& fifoPrintUDec(f,sector,2,10)
			&& fifoPrintString(f,": ")
			&& fifoPrintUDec(f,mm->sectorArrays[ss].sizeK,1,10)
			&& fifoPrintString(f,"kiB @ 0x")
			&& fifoPrintHex(f,mm->addressFlash + offset,8,8)
			&& fifoPrintString(f," (")
			&& fifoPrintUDec(f,(mm->addressFlash + offset)>>10,1,10)
			&& fifoPrintString(f,"ki)\n");
			offset += mm->sectorArrays[ss].sizeK*1024;
			sector++;
		}
	}
	success = success && fifoPrintString(f,"Copy RAM to FLASH block sizes: ");
	for (int b=0; b<ELEMENTS(mm->blockSizes); ++b) {
		if (b!=0) success = success && fifoPrintString(f,", ");
		success = success && fifoPrintUDec(f,mm->blockSizes[b],1,10);
	}
	return success && fifoPrintLn(f);
}

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap17xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, 0x2007C000, 0x20080000 },
	.sectorArrays = { { .sizeK=4, .n=16 }, { .sizeK=32, .n=14 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap2103 = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=4, .n=8 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap2106 = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=16 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap211x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=16 }, {}, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap212x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=8 }, { .sizeK=64, .n=2 }, { .sizeK=8, .n=2},  {} },
	.blockSizes = { 256, 512, 1024, 4096 }	// unchecked
};


//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap213x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=4, .n=8}, { .sizeK=32, .n=14 }, { .sizeK=4, .n=5 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap214x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, 0x7FD00000, },
	.sectorArrays = { { .sizeK=4, .n=8}, { .sizeK=32, .n=14 }, { .sizeK=4, .n=5 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap23xx = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, 0x7FD0C000, 0x7FE00000 },
	.sectorArrays = { { .sizeK=4, .n=8 }, { .sizeK=32, .n=14 }, { .sizeK=4, .n=6 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers17xx[] = {
	&lpcFamilyMember1751,
	&lpcFamilyMember1752,
	&lpcFamilyMember1754,
	&lpcFamilyMember1756,
	&lpcFamilyMember1758,
	&lpcFamilyMember1759,
	&lpcFamilyMember1764,
	&lpcFamilyMember1765,
	&lpcFamilyMember1766,
	&lpcFamilyMember1767,
	&lpcFamilyMember1768,
	&lpcFamilyMember1769,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers210x[] = {
	&lpcFamilyMember2101,
	&lpcFamilyMember2102,
	&lpcFamilyMember2103,
	&lpcFamilyMember2104,
	&lpcFamilyMember2105,
	&lpcFamilyMember2106,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers211x[] = {
	&lpcFamilyMember2114,
	&lpcFamilyMember2124,
	0
};


//SLICE
const LpcFamilyMember* const lpcFamilyMembers213x[] = {
	&lpcFamilyMember2131,
	&lpcFamilyMember2132,
	&lpcFamilyMember2134,
	&lpcFamilyMember2136,
	&lpcFamilyMember2138,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers214x[] = {
	&lpcFamilyMember2141,
	&lpcFamilyMember2142,
	&lpcFamilyMember2144,
	&lpcFamilyMember2146,
	&lpcFamilyMember2148,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers23xx[] = {
	&lpcFamilyMember2364,
	&lpcFamilyMember2365,
	&lpcFamilyMember2366,
	&lpcFamilyMember2367,
	&lpcFamilyMember2368,
	&lpcFamilyMember2377,
	&lpcFamilyMember2378,
	&lpcFamilyMember2378IR,
	&lpcFamilyMember2387,
	&lpcFamilyMember2388,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembersXxxx[] = {
	&lpcFamilyMember1751,
	&lpcFamilyMember1752,
	&lpcFamilyMember1754,
	&lpcFamilyMember1756,
	&lpcFamilyMember1758,
	&lpcFamilyMember1759,
	&lpcFamilyMember1764,
	&lpcFamilyMember1765,
	&lpcFamilyMember1766,
	&lpcFamilyMember1767,
	&lpcFamilyMember1768,
	&lpcFamilyMember1769,

	&lpcFamilyMember2101,
	&lpcFamilyMember2102,
	&lpcFamilyMember2103,
	&lpcFamilyMember2104,
	&lpcFamilyMember2105,
	&lpcFamilyMember2106,

	&lpcFamilyMember2114,
	&lpcFamilyMember2124,

	&lpcFamilyMember2131,
	&lpcFamilyMember2132,
	&lpcFamilyMember2134,
	&lpcFamilyMember2136,
	&lpcFamilyMember2138,

	&lpcFamilyMember2141,
	&lpcFamilyMember2142,
	&lpcFamilyMember2144,
	&lpcFamilyMember2146,
	&lpcFamilyMember2148,

	&lpcFamilyMember2364,
	&lpcFamilyMember2365,
	&lpcFamilyMember2366,
	&lpcFamilyMember2367,
	&lpcFamilyMember2368,
	&lpcFamilyMember2377,
	&lpcFamilyMember2378,
	&lpcFamilyMember2378IR,
	&lpcFamilyMember2387,
	&lpcFamilyMember2388,

	0	// terminator
};

//SLICE
const LpcFamilyMember lpcFamilyMember1751 = {
	.number = 1751,
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1751,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1752 = {
	.number = 1752,
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_1752,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1754 = {
	.number = 1754,
	.sizeFlashK = 128,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1754,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1756 = {
	.number = 1756,
	.sizeFlashK = 256,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1756,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1758 = {
	.number = 1758,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1758,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1759 = {
	.number = 1759,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1759,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};


//SLICE
const LpcFamilyMember lpcFamilyMember1764 = {
	.number = 1764,
	.sizeFlashK = 128,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1764,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1765 = {
	.number = 1765,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1765,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1766 = {
	.number = 1766,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1766,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1767 = {
	.number = 1767,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1767,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1768 = {
	.number = 1768,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1768,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1769 = {
	.number = 1769,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1769,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC210x family
//SLICE
const LpcFamilyMember lpcFamilyMember2101 = {
	.number = 2101,
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_2101,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2102 = {
	.number = 2102,
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_2102,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2103 = {
	.number = 2103,
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2103,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2104 = {
	.number = 2104,
	.sizeFlashK = 120,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2104,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

//SLICE
const LpcFamilyMember lpcFamilyMember2105 = {
	.number = 2105,
	.sizeFlashK = 120,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2105,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

//SLICE
const LpcFamilyMember lpcFamilyMember2106 = {
	.number = 2106,
	.sizeFlashK = 120,
	.sizeRamKs = { 64, },
	.id = LPC_ID_2106,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC211x family

//SLICE
const LpcFamilyMember lpcFamilyMember2114 = {
	.number = 2114,
	.sizeFlashK = 120,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2114,
	.familyMemoryMap = &lpcFamilyMemoryMap211x
};


//SLICE
const LpcFamilyMember lpcFamilyMember2124 = {
	.number = 2124,
	.sizeFlashK = 248,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2124,
	.familyMemoryMap = &lpcFamilyMemoryMap212x
};



////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC213x family

//SLICE
const LpcFamilyMember lpcFamilyMember2131 = {
	.number = 2131,
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2131,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2132 = {
	.number = 2132,
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2132,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2134 = {
	.number = 2134,
	.sizeFlashK = 128,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2134,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2136 = {
	.number = 2136,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2136,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2138 = {
	.number = 2138,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2138,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};


////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC214x family

//SLICE
const LpcFamilyMember lpcFamilyMember2141 = {
	.number = 2141,
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2141,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2142 = {
	.number = 2142,
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2142,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2144 = {
	.number = 2144,
	.sizeFlashK = 128,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2144,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2146 = {
	.number = 2146,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, },
	.id = LPC_ID_2146,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2148 = {
	.number = 2148,
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 8, },
	.id = LPC_ID_2148,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};


////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC23 family

//SLICE
const LpcFamilyMember lpcFamilyMember2364 = {
	.number = 2364,
	.sizeFlashK = 128,
	.sizeRamKs = { 8, 8, 16 },
	.id = LPC_ID_2364,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2365 = {
	.number = 2365,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2365,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2366 = {
	.number = 2366,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2366,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2367 = {
	.number = 2367,
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2367,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2368 = {
	.number = 2368,
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2368,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2377 = {
	.number = 2377,
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2377,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2378 = {
	.number = 2378,
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16, },
	.id = LPC_ID_2378,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2378IR = {
	.number = 2378,
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16, },
	.id = LPC_ID_2378IR,		// initial revision
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2387 = {
	.number = 2387,
	.sizeFlashK = 504,
	.sizeRamKs = { 64, 16, 16, },
	.id = LPC_ID_2387,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2388 = {
	.number = 2388,
	.sizeFlashK = 504,
	.sizeRamKs = { 64, 16, 16, },
	.id = LPC_ID_2388,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

