/*
  lpcMemories.h - NXP LPCxxxx microcontroller memory descriptions. 
  Copyright 2011 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef __lpcMemories_h
#define __lpcMemories_h

/** @file
 * @brief FLASH and RAM map of NXPs LPCxxxx devices.
 *
 * A LPC FLASH consists of multiple sectors of (typically) different sizes. However, only a few sizes are used for
 * a complete FLASH memory, so we store the segmentation as a list of pairs of (size,count).
 */

#include <integers.h>
#include <stdbool.h>
#include <fifoPrint.h>

/** The size of individual sectors within one family is the same - what varies is the total number of sectors
 * implemented in each device. This list of different sizes/counts is terminated by a (0,0) element.
 */
typedef struct __attribute__((packed)) {
	Int8	sizeK;
	Int8	n;
} LpcSectorArray;

typedef struct __attribute__((packed,aligned(4))) {
	Uint32		addressFlash;
	Uint32		addressRams[3];
	LpcSectorArray	sectorArrays[4];
	Uint16		blockSizes[4];	///< Copy RAM to Flash block sizes
} LpcFamilyMemoryMap;

extern LpcFamilyMemoryMap const
	lpcFamilyMemoryMap2103,
	lpcFamilyMemoryMap2106,
	lpcFamilyMemoryMap211x,
	lpcFamilyMemoryMap212x,
	lpcFamilyMemoryMap213x,
	lpcFamilyMemoryMap214x,
	lpcFamilyMemoryMap23xx,
	lpcFamilyMemoryMap17xx,
	lpcFamilyMemoryMap13xx;

typedef struct __attribute__((packed,aligned(4))) {
	Uint32				number;		///< device number, like 1766
	Uint16				sizeFlashK;
	Uint16				sizeRamKs[3];
	Uint32				id;		///< CPU ID
	const LpcFamilyMemoryMap*	familyMemoryMap;
} LpcFamilyMember;

extern const LpcFamilyMember
	lpcFamilyMember1751,
	lpcFamilyMember1752,
	lpcFamilyMember1754,
	lpcFamilyMember1756,
	lpcFamilyMember1758,
	lpcFamilyMember1759,
	lpcFamilyMember1764,
	lpcFamilyMember1765,
	lpcFamilyMember1766,
	lpcFamilyMember1767,
	lpcFamilyMember1768,
	lpcFamilyMember1769,

	lpcFamilyMember2101,
	lpcFamilyMember2102,
	lpcFamilyMember2103,
	lpcFamilyMember2104,
	lpcFamilyMember2105,
	lpcFamilyMember2106,

	lpcFamilyMember2114,
	lpcFamilyMember2124,

	lpcFamilyMember2131,
	lpcFamilyMember2132,
	lpcFamilyMember2134,
	lpcFamilyMember2136,
	lpcFamilyMember2138,

	lpcFamilyMember2141,
	lpcFamilyMember2142,
	lpcFamilyMember2144,
	lpcFamilyMember2146,
	lpcFamilyMember2148,

	lpcFamilyMember2364,
	lpcFamilyMember2365,
	lpcFamilyMember2366,
	lpcFamilyMember2367,
	lpcFamilyMember2368,
	lpcFamilyMember2377,
	lpcFamilyMember2378,
	lpcFamilyMember2378IR,
	lpcFamilyMember2387,
	lpcFamilyMember2388
	;

extern const LpcFamilyMember * const lpcFamilyMembers17xx[];
extern const LpcFamilyMember * const lpcFamilyMembers23xx[];

extern const LpcFamilyMember * const lpcFamilyMembersXxxx[];	///< All known members.

enum {

	LPC_ID_1311	= 0x2C42502B,
	LPC_ID_1313	= 0x2C40102B,
	LPC_ID_1342	= 0x3D01402B,
	LPC_ID_1343	= 0x3D00002B,

	LPC_ID_1769	= 0x26113F37,
	LPC_ID_1768	= 0x26013F37,
	LPC_ID_1767	= 0x26012837,
	LPC_ID_1766	= 0x26013F33,
	LPC_ID_1765	= 0x26013733,
	LPC_ID_1764	= 0x26011922,
	LPC_ID_1759	= 0x25113737,
	LPC_ID_1758	= 0x25013F37,
	LPC_ID_1756	= 0x25011723,
	LPC_ID_1754	= 0x25011722,
	LPC_ID_1752	= 0x25001121,
	LPC_ID_1751	= 0x25001118,

	LPC_ID_2101	= 0,	// same as LPC2103
	LPC_ID_2102	= 0,	// same as PLC2103
	LPC_ID_2103	= 0x0004FF11,

	LPC_ID_2104	= 0xFFF0FF12,
	LPC_ID_2105	= 0xFFF0FF22,
	LPC_ID_2106	= 0xFFF0FF32,

	LPC_ID_2114	= 0x0101FF12,
	LPC_ID_2124	= 0x0101FF13,

	LPC_ID_2131	= 0x0002FF01,
	LPC_ID_2132	= 0x0002FF11,
	LPC_ID_2134	= 0x0002FF12,
	LPC_ID_2136	= 0x0002FF23,
	LPC_ID_2138	= 0x0002FF25,

	LPC_ID_2141	= 0x0402FF01,
	LPC_ID_2142	= 0x0402FF11,
	LPC_ID_2144	= 0x0402FF12,
	LPC_ID_2146	= 0x0402FF23,
	LPC_ID_2148	= 0x0402FF25,

	LPC_ID_2364	= 0x1600F902,
	LPC_ID_2365	= 0x1600E823,
	LPC_ID_2366	= 0x1600F923,
	LPC_ID_2367	= 0x1600E825,
	LPC_ID_2368	= 0x1600F925,
	LPC_ID_2377	= 0x1700E825,
	LPC_ID_2378	= 0x1700FD25,
	LPC_ID_2378IR	= 0x0703FF25,	///< initial revision devices
	LPC_ID_2387	= 0x1800F935,
	LPC_ID_2388	= 0x1800FF35,
};

const LpcFamilyMember* lpcFindById(LpcFamilyMember const * const* list, Uint32 id);
const LpcFamilyMember* lpcFindByNumber(LpcFamilyMember const * const* list, int partNumber);

/** Calculates what sectors are affected from operations within an address range.
 * @param member LPC family member descriptor
 * @param sectorFrom the first sector affected
 * @param sectorTo the last sector affected
 * @param addressFrom the lowest address of the range
 * @param addressTo the highest address of the range.
 * @return true, if the addresses are within the members FLASH range
 */
bool lpcAddressRangeToSectorRange(LpcFamilyMember const *member,
	int *sectorFrom, int *sectorTo, Uint32 addressFrom, Uint32 addressTo);

bool lpcSectorRange(LpcFamilyMember const *member, Uint32 *addressFrom, Uint32 *addressTo, int sector);

/** Finds the sector to a given address.
 * @param member LPC family member descriptor
 * @param address the address of a single byte.
 * @return the sector that contains the byte or -1 if no such sector exists (in that device).
 */
int lpcAddressToSector(LpcFamilyMember const *member, Uint32 address);

/** Finds the number of the highest sector of the whole family.
 */
int lpcFamilySectorHighest(LpcFamilyMemoryMap const *map);

/** Finds the offset from the beginning of the FLASH.
 * @param map the families memory map
 * @param sector any valid sector number within the map or even +1 to calculate FLASH size.
 * @return the starting offset of the given sector.
 */
int lpcFamilySectorOffset(const LpcFamilyMemoryMap* map, int sector);

/** Returns the code used to enable different levels of Code Read Protection (CRP).
 * @param level a protection level, 0..3.
 * @return the values 0xFFFFFFFF, 0x12345678, 0x87654321 and 0x43218765 for levels 0..3.
 */
Uint32 lpcCrpLevelToCode(int level);

/** Translates CRP codes into CRP levels.
 * @param code either a valid CRP code for levels 1..3 or any other value for level 0.
 * @return CRP level 0..3.
 */
int lpcCrpCodeToLevel(Uint32 code);

/** Print all information about a LPCxxxx family member.
 */
bool fifoPrintLpcFamilyMember(Fifo* fifo, LpcFamilyMember const* member);

#endif
