/*
  fifoPopt.h - Program command line parsing using Fifo.
  Copyright 2012-2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef fifoPopt_h
#define fifoPopt_h

/** @file
 * @brief Command-line parsing using Fifo.
 */
#include <integers.h>
#include <fifoq.h>
#include <fifoParseStructure.h>

typedef struct {
	Int32	a;
	Int32	b;
} PairInt32;

typedef struct {
	char		shortOption;	///< Option character (following the '-').
	bool*		value;		///< where to write the parsed result?
} FifoPoptBool;


typedef struct {
	char		shortOption;	///< Option character (following the '-').
	Int32*		value;		///< where to write the indicator?
	Int32		indicator;	///< value is set to this if option is found.
} FifoPoptInt32Flag;


typedef bool (*FifoParseInt32)(Fifo*, Int32*);

typedef struct {
	char		shortOption;	///< Option character (following the '-').
	Int32*		value;		///< where to write the parsed result?
	FifoParseInt32	parseInt;	///< integer parser to apply, 0 = parseIntCStyle
} FifoPoptInt32;


typedef struct {
	char		shortOption;	///< Option character (following the '-').
	Fifo*		value;		///< where to write the parsed result?
} FifoPoptString;

typedef struct {
	char		shortOption;	///< Option character (following the '-').
	PairInt32*	value;		///< where to write the parsed result?
	const char*	separator;	///< range operator symbol, 0 = ..
	FifoParseInt32	parseInt;	///< integer parser to apply, 0 = parseIntCStyle
} FifoPoptInt32Range;

typedef struct {
	char			shortOption;	///< Option character (following the '-').
	int*			value;		///< where to write the parsed result?
	const char* const*	symbols;	///< array of literal symbols
} FifoPoptSymbol;

typedef struct {
	char		shortOption;	///< Option character
	int		n;		///< number of elements
	Int32*		values;		///< array of results
	const char*	separator;	///< separator, typically ","
	FifoParseInt32	parseInt;	///< integer parser to apply, 0 = parseIntCStyle
} FifoPoptInt32Tuple;

/** Scans for an option (short or long) without modifying the command line.
 * This is intended for the help option.
 * @param cmdLine the accumulated command line list.
 * @param shortOption the option character of the short option, e.g. '?' for -? .
 * @param longOption the option string of the long option, e.b. 'help' for --help .
 * @return true, if the option is found, false otherwise.
 */
bool fifoPoptScanOption(Fifoq *cmdLine, char shortOption, const char* longOption);

/** Accumulates up a processes' command line into a single Fifo.
 * @param argList a Fifo for accumulating the parameter list.
 * @param argc the main()'s argc parameter
 * @param argv the main()'s argv parameter
 * @return true if all data fit into argList.
 */
bool fifoPoptAccumulateCommandLine(Fifoq *argList, int argc, const char* const *argv);

/** Parses accumulated options immediately (and in direct sequence) available. Only flag options can be
 * accumulated.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptBool(Fifoq *cmdLine, const FifoPoptBool *options);

/** Parses accumulated options immediately (and in direct sequence) available. Only flag options can be
 * accumulated.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found.
 */
bool fifoPoptInt32Flag(Fifoq *cmdLine, const FifoPoptInt32Flag *options);

/** Parses an int option immediately available.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptInt32(Fifoq *cmdLine, const FifoPoptInt32 *options);

/** Parses all string options immediately (and in direct sequence) available.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptString(Fifoq *cmdLine, const FifoPoptString *options);

/** Extracts a non-option from the command line and puts it into the list of non-options. A 0-character is appended
 * after the non-option string to separate it from following strings. It should be noted, that the output fifo can be
 * derived from the input Fifo in such a way, that they share buffers and this function does an in-place update. For
 * that, the output Fifo must have the wPos set to the input's Fifo buffer start and wTotal=size and rTotal=0.
 * @param cmdLine a command line.
 * @param listOfNonOptions an (initialized) fifo for holding the non-option string list.
 * @return true, unless the output Fifo overflows.
 */
bool fifoPoptNonOptionAccumulate(Fifoq *cmdLine, Fifoq *listOfNonOptions);

/** Parses an int range immediately available.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptInt32Range(Fifoq *cmdLine, const FifoPoptInt32Range *options);

/** Parses an option of a fixed size tuple of ints.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptInt32Tuple(Fifoq *cmdLine, const FifoPoptInt32Tuple *options);

/** Parses a symbolic option immediately available.
 * @param cmdLine a command line with the options separated by 0-characters.
 * @param options a zero-terminated array of option descriptors.
 * @return true if option found and parsed correctly.
 */
bool fifoPoptSymbol(Fifoq *cmdLine, const FifoPoptSymbol* options);

#endif
