#ifndef int32Int32Integrator_h
#define int32Int32Integrator_h

#include <integers.h>
#include <int32Math.h>
#include <int64Math.h>

/** @file
 * @brief Toolkit for doing numerical integration in a sampled system with mostly 32bit operations.
 *
 * This module is intended for 32bit CPUs. 64bit divide is used in initialization functions only.
 * All other functions use 64bit shifts at most. This allows good performance on 32bit systems.
 *
 * Int32Scale allows precision scaling of values, for example scaling by 1/fsample in the numerical int32Integrator.
 * Keep in mind, that the values may well be E10 or E20 fixed point values instead of integers without a change in
 * function calls.
 *
 * Int32Integrator performs numerical integration. Every call to integrate(v) adds a small portion that adds up to v
 * after n steps, where scale(n) = 1.
 */

/** Object for scaling int32-values with high precision.
 */
typedef struct {
	Int32	m;		///< full integer range
	Int8	e;		///< range 0..31
} Int32Scale;

/** Fixed point value: 32bits integral part, 32bit fractional part.
 * Atomic access to value OR fraction, but never both.
 */
typedef union {
	struct {
		Uint32	fraction;
		Int32	value;
	};
	Int64		value_e32;
} Q3232;

/** Initializes a scaler with a ratio p/q. This is an approximation. Do not expect scaling to be exact.
 * @param scale the opject to initialize.
 * @param mul the multiplier p
 * @param div the divider q
 */
inline static void int32ScaleInit(Int32Scale *scale, int mul, int div) {
	const int eF = 31 + uint32Log2Floor(div) - uint32Log2Ceil(mul);
	scale->m = int64Div((Int64)mul<<eF,div);
	scale->e = eF;
}

/** Scales an integral value, resulting in a Q32.32 value.
 * @param scale the scaler object
 * @param value an integral / Exx value withing the Int32 range.
 * @return the scaled value as a pair or integral / Exx part and the fractional part of that value. Normally the
 *   fractional part is discarded, because of its over-preciseness.
 */
inline static Q3232 int32ScaleValue(const Int32Scale *scale, Int32 value) {
	Q3232 p = {
		{
			.value = (Int64)value * scale->m >> scale->e,
			.fraction = int64MulExp2((Int64)value * scale->m,32 - scale->e)
		}
	};
	return p;
}

/* Scales forward the fraction of a value.
inline static Int32 int32ScaleFraction(const Int32Scale *s, Int32 fraction) {
	return int64MulExp2( fraction, 32 - s->e) * s->m;
}
*/

/** Integrated value in a sampled system. Values are considered for times t=0,1,... .
 * Sample frequency is chosen by the scale object. A scale object of 1/fsample is used to model a sampling
 * frequency of fsample.
 */
typedef struct {
	const Int32Scale *	time;		///< scales by 1/f = T = sample distance 
	Q3232			q;		///< the current value and fraction
} Int32Integrator;

/** Performs one single step of numerical integration (summing up).
 * @param i the integrator object.
 * @param dxdt the speed.
 */
inline static void int32IntegratorI(Int32Integrator *i, Int32 dxdt) {
	const Q3232 v = int32ScaleValue(i->time,dxdt);
	i->q.value_e32 += v.value_e32;
}

/** Movement with limited dx/dt. This is a handy method for calculating the position of mass that should finally end up
 * at the destination position, but is limited in speed to go there.
 * @param i the integrator object.
 * @param dxdtMin the minimum speed - meant to be the maximum speed backwards.
 * @param dxdtMax the maximum speed forward.
 * @param xTo the destination position, integral/Exx part.
 */
void int32IntegratorMove(Int32Integrator *i, Int32 dxdtMin, Int32 dxdtMax, Int32 xTo);

/** Returns the current integrator value.
 * @param i the integrator object.
 * @return the current position, integral/Exx part.
 */
inline static Int32 int32IntegratorValue(Int32Integrator *i) {
	return i->q.value;
}

#endif

