//HEADER
/*
  int64Math-sliced.c 
  Copyright 2012-2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#include <int64Math.h>

//SLICE
Uint64 uint64Div(Uint64 a, Uint64 b) {
	if (b==0) return 0;

	int lz = 0;
	for (int i=0; i<64 && ((b<<lz & 1llu<<63)==0); ++i) lz=i;

	Uint64 result = 0;
	Uint64 aa = a;
	for (int i=lz; i>=0; --i) {
		if (aa >= (b<<i)) {
			aa -= b<<i;
			result |= 1<<i;
		}
	}
	return result;
}

//SLICE
#ifdef NATIVE_64BIT
// nothing
#else
Uint64 uint64Mod(Uint64 a, Uint64 b) {
	return a - b*uint64Div(a,b);
}
#endif

//SLICE
#ifdef NATIVE_64BIT
// nothing
#else
Int64 int64Div(Int64 a, Int64 b) {
	int sign = 1;
	if (a<0) {
		sign *= -1;
		a = -a;
	}
	if (b<0) {
		sign *= -1;
		b = -b;
	}
	return ((Int64)uint64Div(a,b))*sign;
}
#endif

//SLICE
Int64 int64Mod(Int64 a, Int64 b) {
	return a - b*int64Div(a,b);
}

//SLICE
Int64 int64SqrtFloor(Int64 x) {
	if (x<=0) return x;	// catches x=0, too

	Int64 r = 1, rr = 0;
	while (r!=rr) {
		rr = r;
		r = (r + uint64Div(x,r) + 1) >> 1;	
	}
	return r*r<x ? r : r-1;
}

//SLICE
Int32 int32LinearExtrapolate(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x) {
	return int64Div( int64RoundForDivide((Int64)y0*(x1-x) + (Int64)y1*(x-x0),(x1-x0)), (x1-x0) );
}

//SLICE
Int32 int32LinearInterpolate(Int32 x0, Int32 y0, Int32 x1, Int32 y1, Int32 x) {
	if (x<x0) return y0;
	else if (x>x1) return y1;
	else return int32LinearExtrapolate(x0,y0,x1,y1,x);
}

