/*
  seqio.h 
  Copyright 2011 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef __seqio_h
#define __seqio_h

/** @file
 * @brief Sequential IO consisting of multiple data blocks mainly for I2C communications.
 *
 * A Seqio object is a buffer between a user program an a device driver in an asynchronous IO scenario. The user
 * program defines the IO transaction. The device driver extracts the bytes to send from the Seqio write buffer and
 * stores the received bytes in the Seqio read buffer. After IO completion the user program can extract the received
 * bytes from the Seqio read buffer.
 * An IO transaction consists of one or more reads and/or writes, each of which consists of one or more bytes.
 * A sequence of consecutive writes can be joined to a single write unless explicit bounds are included. The same is
 * true for sequences of reads. Any transfer marked with the BOUND flag will not be joined with the preceding transfer.
 * A read transfer can never be joined with a write transfer and vice versa.
 * Seqio implements data transfer in both directions in a half duplex mode. At a given time a read transfer can be
 * active or a write transfer but not both.
 * Seqio targets mainly at 32bit systems (32bit integers and 32bit pointers). 64bit systems will waste a few bytes.
 */

#include <integers.h>
#include <stdbool.h>

#if 0
typedef union {
	int	i;
	char*	p;
} SeqioElement;
#else
typedef Int SeqioElement;
#endif

/** The encoding of the seqio action sequence: reading/writing immediate or indirect data, explicit bounds.
 */
enum {
	// implemenentation secrets - do not use in user code...
	SEQIO_W_BIT		=0x10000000,	///< all write operations have it, others don't
	SEQIO_R_BIT		=0x20000000,	///< all read operations have it, others don't
	SEQIO_P_BIT		=0x40000000,	///< IMPORTANT: every OP on a buffer has it, all others don't
	SEQIO_B_BIT		=0x80000000,	///< marks a block bound within multiple reads or writes
	SEQIO_L_SHIFT		=24,		///< length position for immediate read/write (1..3)
	SEQIO_L_MASK		=0x00FFFFFF,	///< length for block reads/writes can be large

	// public constants
	SEQIO_END		=0,		///< Use this to terminate a sequence.
	SEQIO_WRITE		=SEQIO_W_BIT | 1<<SEQIO_L_SHIFT,	///< write 1 immediate byte
	SEQIO_WRITE2		=SEQIO_W_BIT | 2<<SEQIO_L_SHIFT,	///< write 2 immediate bytes
	SEQIO_WRITE3		=SEQIO_W_BIT | 3<<SEQIO_L_SHIFT,	///< write 3 immediate bytes
	SEQIO_READ		=SEQIO_R_BIT | 1<<SEQIO_L_SHIFT,	///< driver shall read 1 immediate byte
	SEQIO_READ2		=SEQIO_R_BIT | 2<<SEQIO_L_SHIFT,	///< driver shall read 2 immediate bytes
	SEQIO_READ3		=SEQIO_R_BIT | 3<<SEQIO_L_SHIFT,	///< driver shall read 3 immediate bytes
	SEQIO_WRITE_BUFFER	=SEQIO_W_BIT | SEQIO_P_BIT,		///< write memory block
	SEQIO_READ_BUFFER	=SEQIO_R_BIT | SEQIO_P_BIT,		///< driver shall read memory block
	SEQIO_BOUND		=SEQIO_B_BIT,				///< perform repeated start in I2C
};

typedef enum {
	SEQIO_PREPARE = 0,	///< sequence is being prepared
	SEQIO_ACTION,		///< sequence prepared and now ready for being processed by driver
	SEQIO_PENDING,		///< driver has not yet confirmed all bytes sent
	SEQIO_DONE,		///< driver finished with sequence successfully
	SEQIO_DONE2,
	SEQIO_ERROR,		///< driver finished with sequence with error
	SEQIO_ERROR2,
} SeqioStatus;

typedef struct {
	SeqioElement*	elements;	///< IO sequence
	Uint16		size;		///< byte size of the elements array
	Uint16		position;	///< element position
	Uint16 volatile	status;		///< See SeqioStatus
	Uint16		charPosition;	///< character position while processing
} Seqio;

/** Resets Seqio for preparation.
 */
void seqioReset(Seqio *seqio);

// preparing: interface of user code
/** Writes 1 to 3 immediate bytes.
 * @param seqio the sequential IO object.
 * @param cs the data to write in little endian order.
 * @param n a value between 1 and 3
 * @param bound true to mark this write as a block, false to (potentially) merge the write with others.
 * @return true if successful, false if buffer overflow or parameter error.
 */
bool seqioPrepareWrite(Seqio *seqio, int cs, int n, bool bound);

/** Writes data from a buffer.
 * @param seqio the sequential IO object.
 * @param data the pointer to the write data. Write data must exist until the IO transaction is finished.
 * @param n data size. Zero bytes are not allowed.
 * @param bound true to mark this write as a block, false to (potentially) merge the write with others.
 */
bool seqioPrepareWriteBlock(Seqio *seqio, const void *data, int n, bool bound);

/** Reads 1 to 3 bytes.
 * @param seqio the sequential IO object.
 * @param n the number of butes to read (later), range = 1..3
 * @param bound true to mark this read as a block, false to (potentially) merge the read with others.
 */
bool seqioPrepareRead(Seqio *seqio, int n, bool bound);

/** Reads up to 3 bytes.
 * @param seqio the sequential IO object.
 * @param data the buffer for storing the bytes to read.
 * @param n the number of butes to read (later), range = 1..3
 * @param bound true to mark this read as a block, false to (potentially) merge the read with others.
 */
bool seqioPrepareReadBlock(Seqio *seqio, void *data, int n, bool bound);

/** Marks the current position as end of IO. The state transition to SEQIO_ACTION and a device driver may access the
 * IO objects from this point.
 * @param seqio the sequential IO object.
 * @return false if the IO object was not in state SEQIO_PREPARE.
 */
bool seqioPrepareEnd(Seqio *seqio);

/** Prepares an IO object for replay. It must be prepared successfully before.
 */
bool seqioPrepareReplay(Seqio *seqio);

/** Checks, if the device driver has finished the transaction. Can be called at any time.
 * @return true, if user code can access the IO object for reading/error checking.
 */
bool seqioIsDone(const Seqio *seqio);

/** Checks, if the device driver has finished the transaction, exactly how seqioIsDone() does.
 * However, this function returns true only once - it performs edge detection.
 * @return true, if user code can access the IO object for reading/error checking.
 */
bool seqioIsJustDone(Seqio *seqio);

/** Checks for success after the IO is completed. Can be called at any time.
 * @return true if all bytes have been transmitted/received or false if anything went wrong.
 */
bool seqioIsSuccessful(const Seqio *seqio);

// reading of acquired data: interface of user code
/** Checks if more bytes are available for reading. Must not be called until a device driver finished successfully
 * with this IO object.
 * @return true if at least one more byte can be read.
 */
bool seqioCanRead(Seqio *seqio);

/** Reads one byte. Must not be called unless seqioCanRead() returns true.
 */
char seqioRead(Seqio *seqio);

////////////////////////////////////////////////////////////////////////////////////////////////////
// processing: interface to IO driver

/** Defines what a device driver is supposed to do next.
 * The following constants are bits that need to be ORed together to form suitable action descriptions. This means
 * that if-else conditions are more suitable than switch statements as more than one combination may occur.
 */
typedef enum {
	SEQIO_ACTION_READ	=0x01,	///< read action
	SEQIO_ACTION_WRITE	=0x02,	///< write action
	SEQIO_ACTION_FIRST_BYTE	=0x04,	///< first byte of a read/write action
	SEQIO_ACTION_LAST_BYTE	=0x08,	///< last byte of a read/write action
	SEQIO_ACTION_END	=0x00,	///< completed - either with or without success
} SeqioAction;

/** Checks, what to do next.
 */
SeqioAction seqioAction(const Seqio *seqio);

/** Reads a single character from the IO object buffer. This character must be transmitted by the hardware.
 */
char seqioActionRead(Seqio *seqio);

/** Writes a single character to the IO object buffer. This character was received by the hardware and will be
 * read by the user program from the IO object buffer.
 */
void seqioActionWrite(Seqio *seqio, char c);

/** Confirms, that the last byte read from Seqio is successfully sent by the hardware. This function triggers the
 * transition from SEQIO_PENDING to SEQIO_END. It must be called only in SEQIO_PENDING state.
 */
void seqioActionConfirm(Seqio *seqio);

/** Indicates abnormal termination of the IO operation when called by the device driver.
 */
void seqioActionError(Seqio *seqio);

#endif

