/*
  isp.c - main program of isp, a NXP LPC ARM controller isp programmer for Linux.
  Copyright 2011-2013 Marc Prager
 
  isp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  isp is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with isp.
  If not see <http://www.gnu.org/licenses/>
 */

// C standard
#include <fcntl.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>

// *nix specific
#include <termios.h>
#include <unistd.h>
#include <time.h>		// high resolution timers

// c-any library headers
#include <fifoParse.h>
#include <fifoParseStructure.h>
#include <fifoPopt.h>
#include <fifoPrint.h>
#include <lpcMemories.h>
#include <uu.h>
#include <macros.h>
#include <int32Math.h>

// my own linux-specific functions
#include <c-linux/serial.h>
#include <c-linux/hexFile.h>

/* History:
 * Version 1.0 (09/2011)
 * ======================
 * - full functionality
 *
 * Version 1.1 (04/2012)
 * ======================
 * - added Intel hex-file support, although in a not very clean programming style.
 * - put under GPL (03/2013)
 *
 * Version 2.0 (07/2013)
 * ======================
 * - added support for LPC800 (binary ISP protocol)
 * - clean up of input functions
 * - changed options, many new options.
 * - added execute by RESET as universal start code method.
 * - added verbosity level selection.
 * - improved scripting versality by adding silencing options -z and -Z.
 * - added detailed debugging output
 * - added flow control functionality
 * LPC800 NO_ISP mode NOT supported.
 */

#define ISP_VERSION_COPYRIGHT "2.0 ($Rev: 71 $) (C) Marc Prager 2011-2013."
#define ISP "isp"

static inline bool lpcReset(int fd, bool active) {
	return serialSetDtr(fd,active);
}

static inline bool lpcBsl(int fd, bool active) {
	return serialSetRts(fd,active);
}

static char outBuffer[4096];
static char inBuffer[1000];
Fifo fifoOut = { outBuffer, sizeof outBuffer };
Fifo fifoIn = { inBuffer, sizeof inBuffer };

bool lineChar = false;	// last char, that was inserted into fifoIn was one of the line separators.

static bool useEcho = false;
static int subBlockSize = 64;		// for binary protocol only

const char* escape(char c) {
	//static char buffer[5];
	static char buffer[20];
	if (c=='\r') return "\\r";
	else if (c=='\n') return "\\n";
	else if (0x20<=c && c<=127) {
		buffer[0] = c;
		buffer[1] = 0;
		return buffer;
	}
	else {
		sprintf(buffer,"\\x%02X",c & 0xFF);
		return buffer;
	}
}

#include <stdarg.h>
/** This function writes to stdout for result processing (program output).
 * It is not subject to verbosity level or message redirection. This is
 * stdout.
 */
int printfOut(const char *format, ...) {
	va_list args;
	va_start(args,format);
	const int n = vprintf(format,args);
	fflush(stdout);
	va_end(args);
	return n;
}

enum Verbosity {
	VERBOSITY_SILENT_ERROR,
	VERBOSITY_SILENT,
	VERBOSITY_NORMAL,
	VERBOSITY_DEBUG,
};

static Int32 verbosity = VERBOSITY_NORMAL;

/** Prints out debug messages not normally expected by the user.
 * Subject to redirection, normally on cerr.
 */
int printfDebug(const char *format, ...);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int printfNormal(const char *format, ...);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int vprintfNormal(const char *format, va_list args);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int printfWarning(const char *format, ...);

/** Prints out error messages normally expected by the user in case of a malfunction.
 * Subject to redirection, normally on cerr.
 */
int printfError(const char *format, ...);

////////////////////////////////////////////////////////////////////////////////////////////////////

enum {
	FIFO_OUT_MAX = 80,
};
// Useful error output diagnostics
// debugging output, 8-bit human readable.
void printfDebugFifo(Fifo fifo) {
	char c;
	for (int l=0; l<FIFO_OUT_MAX && fifoCanRead(&fifo); )
		l += printfDebug("%s",escape(fifoRead(&fifo)));
	if (fifoCanRead(&fifo)) printfDebug("\\[..%d more]",(int)fifoCanRead(&fifo));
}

void printfErrorFifo(Fifo fifo) {
	char c;
	for (int l=0; l<FIFO_OUT_MAX && fifoCanRead(&fifo); )
		l += printfError("%s",escape(fifoRead(&fifo)));
	if (fifoCanRead(&fifo)) printfError("\\[..%d more]",(int)fifoCanRead(&fifo));
}

// data output
// data output
void printFifo(FILE *dst, Fifo *fifo) {
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		fprintf(dst,"%c",c);
	}
}

// data output
bool ioWriteFifo(int fd, Fifo *fifo) {
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		if (1!=write(fd,&c,1)) {
			printfError("ERROR: IO failed on descriptor %d.\n",fd);
			return false;
		}
	}
	return true;
}

const char* returnMessage(int returnCode) {
	switch(returnCode) {
		case 0:		return "CMD_SUCCESS";
		case 1:		return "INVALID_COMMAND";
		case 2:		return "SRC_ADDR_ERROR";
		case 3:		return "DST_ADDR_ERROR";
		case 4:		return "SRC_ADDR_NOT_MAPPED";
		case 5:		return "DST_ADDR_NOT_MAPPED";
		case 6:		return "COUNT_ERROR";
		case 7:		return "INVALID_SECTOR";
		case 8:		return "SECTOR_NOT_BLANK";
		case 9:		return "SECTOR_NOT_PREPARED";
		case 10:	return "COMPARE_ERROR";
		case 11:	return "BUSY";
		case 12:	return "PARAM_ERROR";
		case 13:	return "ADDR_ERROR";
		case 14:	return "ADDR_NOT_MAPPED";
		case 15:	return "CMD_LOCKED";
		case 16:	return "INVALID_CODE";
		case 17:	return "INVALID_BAUD_RATE";
		case 18:	return "INVALID_STOP_BIT";
		case 19:	return "CODE_READ_PROTECTION";
		default:	return "UNDEFINED ERROR";
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Flow control hook


////////////////////////////////////////////////////////////////////////////////////////////////////

/** Provides a point for interrupting the program an let the user take control
 * @return true to continue, false to stop program
 */
bool flowControlHook(char command, const char* format, ...);

/** This function reads in the answer from the LPC into fifoIn.
 * It quickly reads in all characters up to an CR or LF. CR and LF are both represented by LF in the Fifo.
 * Two LF in direct succession are avoided. If no CR or LF is found, this function runs into a timeout that should
 * terminate this simple application.
 * @return true, if a line was read, false if timeout happened or buffer was too small.
 */
bool fifoLoadLine(int fd, Fifo *fifoIn) {
	char c;
	while (1==read(fd,&c,1)) {
		switch(c) {
			case 0x13:	// flow control: stop
			case 0x11:	// flow control: start
				printfError("ERROR: Flow control not implemented :-(\n");
				return false;
			case '\n':
			case '\r':
				if (!lineChar) {
					lineChar = true;
					if (fifoPrintChar(fifoIn,'\n')) return true;
					else {
						printfError("ERROR: input buffer overflow.\n");
						return false;
					}
				}
				else ; // discard line separators in succession
				break;
			default:
				if (!fifoPrintChar(fifoIn,c)) return false;
				lineChar = false;
		}
	}
	return false;	// timeout
}

/** Reads in another line, only if fifo does not already contain one.
 */
bool fifoLoadLineOnDemand(int fd, Fifo *fifo) {
	// check, if we need reloading...
	if (!fifoContainsChar(fifo,'\n')) {
		// maybe we should throw away the current contents??

		if (fifoLoadLine(fd,&fifoIn)) {
			printfDebug("lpc:");
			printfDebugFifo(fifoIn);
			return true;
		}
		else {
			printfError("ERROR: no answer from lpc.\n");
			return false;
		}
	}
	else return true;	// already a line in Fifo - nothing to do
}


/** This function reads in the fixed-size answer from the LPC into fifoIn.
 * If not enough characters are received, this function runs into a timeout that should terminate this simple
 * application.
 * @param n the number of bytes to receive.
 * @return true, if n characters where read, false if timeout happened.
 */
bool fifoLoadN(int fd, Fifo *fifoIn, int n) {
	char c;
	for (int i=0; i<n; i++) {
		if (1==read(fd,&c,1)) {
			lineChar = false;	// it's considered binary data.
			// there cannot be character-based flow control
			if (!fifoPrintChar(fifoIn,c)) {
				printfError("ERROR: Input buffer overflow.");
				return false;
			}
		}
		else {
			printfError("ERROR: fifoLoadN(): serial timeout. requested=%d, received=%d\n",n,i);
			return false;
		}
	}
	return true;
}

/** As LPC ISP programs are all bull-shit, we scan for the expected string within all the garbage instead of expecting
 * an EXACT answer. If it's found, the whole line is removed from the input.
 */
bool fifoFindStringInLine(Fifo *input, const char *word) {
	Fifo clone = *input;
	if (fifoMatchUntilPattern(&clone,word) && fifoMatchUntilChar(&clone,'\n')) {
		fifoCopyReadPosition(input,&clone);
		return true;
	}
	else return false;
}

/** Reads in a line (if neccessary) that is not an echo line.
 * This function priorizes the echo!!
 */
bool lpcGetNonEchoAnswerLine(int fd, Fifo *fifo, const char *echo) {
	// check, if we need reloading...
	if (!fifoLoadLineOnDemand(fd,fifo)) return false;

	Fifo clone = *fifo;
	// check if it's an echo
	if (fifoFindStringInLine(fifo,echo)) {
		// yes it is, but it's already removed now.
		printfDebug(" - skipping echo line. (pattern was \"%s\", Fifo was \"",echo);
		printfDebugFifo(clone);
		printfDebug("\")\n");

		// load another, if neccessary...
		return fifoLoadLineOnDemand(fd,fifo);
	}
	else return true;	// not an echo
}

/** Consumes a line along with anything (left) in it.
 */
bool fifoConsumeLine(Fifo *fifo) {
	return fifoMatchUntilChar(fifo,'\n');
}

/** Reads one Int32 from a line and consumes the whole line, discarding anything else.
 */
bool fifoFindInt(Fifo *fifo, Int32 *result) {
	return	fifoParseInt(fifo,result)
		&& fifoConsumeLine(fifo);
}

/** Reads one Uint32 from a line and consumes the whole line, discarding anything else.
 */
bool fifoFindUnsigned(Fifo *fifo, Uint32 *result) {
	return	fifoParseUnsigned(fifo,result)
		&& fifoConsumeLine(fifo);
}

/** Reads the result code from a (zero return parameters) command.
 * Optionally the echo of LPC is skipped. The echo should be a pattern of the sent command.
 */
bool lpcFindResult(int fd, Fifo *fifo, const char* echo, Uint32 *result) {
	Fifo cloneBefore = *fifo;
	if (lpcGetNonEchoAnswerLine(fd,fifo,echo)
	&& fifoFindUnsigned(fifo,result)) {
		printfDebug(" - return code %s\n",returnMessage(*result));

		if (*result==0) return true;
		else {
			printfError("ERROR: %s\n",returnMessage(*result));
			return false;
		}
	}
	else {
		printfError("ERROR: could not read return code.\n");
		printfError("Input fifo before line load was \"");
		printfErrorFifo(cloneBefore);
		printfError("\"\nInput fifo after line load was \"");
		printfErrorFifo(*fifo);
		printfError("\"\n");
		return false;
	}
}

/** Reads n integer values after the result code was removed.
 */
bool lpcFindUnsignedValues(int fd, Fifo *fifo, Uint32 *values, int n) {
	for (int i=0; i<n; i++) {
		if (fifoLoadLineOnDemand(fd,fifo)
		&& fifoFindUnsigned(fifo,&values[i])) { // fine
			printfDebug(" - return value[%d]\n",i);
		}
		else {
			printfError("ERROR: could not read return value[%d]\n",i);
			return false;
		}
	}
	return true;
}

bool lpcWriteFifo(int fd, Fifo *fifo) {
	printfDebug(ISP ":");
	printfDebugFifo(*fifo);
	printfDebug("\n");
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		flowControlHook('1',"1 single byte '%s' send.",escape(c));
		if (1!=write(fd,&c,1)) {
			printfError("ERROR: Timeout.\n");
			return false;
		}
	}
	return true;
}

bool lpcWriteString(int fd, const char *data) {
	if (fifoPrintString(&fifoOut,data)) return lpcWriteFifo(fd,&fifoOut);
	else {
		printfError("ERROR: Output buffer overflow.\n");
		return false;
	}
}

/** Maximum detailed diagnostics.
 */
bool lpcSync(int fd, int crystalHz) {
	if (!lpcWriteString(fd,"?")
	|| !fifoLoadLineOnDemand(fd,&fifoIn)) return false;

	if (fifoFindStringInLine(&fifoIn,"Synchronized")) {
		printfDebug(" - Accepted by isp\n");
	}
	else {
		printfError("ERROR: Synchronization failed.\n");
		return false;
	}

	if (!lpcWriteString(fd,"Synchronized\r\n")
	|| !fifoLoadLineOnDemand(fd,&fifoIn)) return false;

	// check, if we get something like "bla Synchronized bla\n"
	if (fifoFindStringInLine(&fifoIn,"Synchronized")) {
		printfDebug(" - Accepted by isp, but needs one more line with OK\n");
	}
	else {
		printfError("ERROR: Invalid answer.\n");
		return false;
	}

	if (!fifoLoadLineOnDemand(fd,&fifoIn)) return false;

	if (fifoFindStringInLine(&fifoIn,"OK")) {
		printfDebug(" - Accepted by isp\n");
		
	}
	else {
		printfError("ERROR: Invalid answer.\n");
		return false;
	}

	if (crystalHz > 0) {
		const Uint32 param = (crystalHz+500)/1000;
		fifoPrintUDec(&fifoOut,param,1,10); fifoPrintString(&fifoOut,"\r\n");
		Fifo clone = fifoOut;
		if (!lpcWriteFifo(fd,&fifoOut)) return false;

		if (!fifoLoadLineOnDemand(fd,&fifoIn)) return false;
		Uint32 result;
		if (fifoFindUnsigned(&fifoIn,&result)) {
			printfDebug(" - crystal frequency reply.\n");
		}
		else return false;

		if (result!=param) {
			printfError("ERROR: echo frequency mismatch.\n");
			return false;	// consume line containing crystal value
		}

		if (!fifoLoadLineOnDemand(fd,&fifoIn)) return false;
		if (fifoFindStringInLine(&fifoIn,"OK")) {
			printfDebug(" - Accepted by isp\n");
			
		}
		else {
			printfError("ERROR: Invalid answer.\n");
			return false;
		}
	}

	return true;
}

bool lpcSendAndCheck(int fd, Fifo *fifoIn, const char *echo) {
	Uint32 result;
	bool ioSuccess =
		fifoPrintString(&fifoOut,"\r\n")
		&& lpcWriteFifo(fd,&fifoOut)
		&& lpcFindResult(fd,fifoIn,echo,&result);

	if (ioSuccess) {
		if (result==0) return true;
		else {
			printfError("ERROR: %s\n",returnMessage(result));
			return false;
		}
	}
	else return false;	// error message already provided
}


/** A command with no integer argument and n result integers.
 */
bool lpcCommand2Unsigneds(int fd, Fifo *fifoIn, char commandChar, Uint32 *values, int n) {
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern)
		&& lpcFindUnsignedValues(fd,fifoIn,values,n);
}

/** A command with 1 integer argument and zero return values.
 */
bool lpcCommandUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a) {
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);
}

/** A command with 2 integer arguments and zero return values.
 */
bool lpcCommandUnsignedUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a, Uint32 b) {
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,b,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);

}

/** A command with 3 integer arguments and zero return values.
 */
bool lpcCommandUnsignedUnsignedUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a, Uint32 b, Uint32 c) {
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,b,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,c,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);

}

bool lpcBaud(int fd, int baud, int stopBits) {
	if (!flowControlHook('B',"B set baud rate to %d, %d stop bits.",baud,stopBits)) return false;
	return lpcCommandUnsignedUnsigned(fd,&fifoIn,'B',baud,stopBits);
}

// At least LPC17 answers A 0\r0\r\n on A 0\r\n. A \n is missing.
// Therefore we use only \r in this command.
// We turn echo off quickly to get rid of this shit.
bool lpcEcho(int fd, Fifo *fifoIn, bool on) {
	return lpcCommandUnsigned(fd,fifoIn,'A',(int)on);	
}

bool lpcReadBootCodeVersion(int fd, Fifo *fifoIn, Uint32 *version) {
	Uint32 values[2];
	if (!flowControlHook('K',"K read boot code version.")) return false;
	if (lpcCommand2Unsigneds(fd,fifoIn,'K',values,2)) {
		printfDebug("Boot code %u.%u\n",values[1],values[0]);
		*version = values[1]<<8 | values[0];
		return true;
	}
	else return false;
}

bool lpcReadPartId(int fd, Fifo *fifoIn, Uint32 *partId) {
	if (!flowControlHook('J',"J read part ID.")) return false;
	if (lpcCommand2Unsigneds(fd,fifoIn,'J',partId,1)) {
		printfDebug("Part ID 0x%08X\n",*partId);
		return true;
	}
	else return false;
}

//Reads out the device serial number. @param sn 4 32-bit integers.
bool lpcReadUid(int fd, Fifo *fifoIn, Uint32 *uid) {
	if (!flowControlHook('N',"N read device UID.")) return false;
	return	lpcCommand2Unsigneds(fd,fifoIn,'N',uid,4);
}

bool lpcCopyRamToFlash(int fd, Fifo *fifoIn, unsigned addressFlash, unsigned addressRam, int n) {
	if (!flowControlHook('C',"C copy RAM 0x%08X to FLASH 0x%08X, size=%d.",addressRam,addressFlash,n)) return false;
	return	lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'C',addressFlash,addressRam,n);
}

bool lpcGo(int fd, Fifo *fifoIn, Uint32 pc, bool thumbMode) {
	const char echo[] = { 'G', 0 };

	if (!flowControlHook('G',"G execute address 0x%08X, mode %c.",pc,thumbMode ? 'T' : 'A')) return false;
	return	fifoPrintString(&fifoOut,"G ")
		&& fifoPrintUDec(&fifoOut,pc,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintChar(&fifoOut,thumbMode ? 'T' : 'A')
		&& lpcSendAndCheck(fd,fifoIn,echo);
}

bool lpcPrepareForWrite(int fd, Fifo *fifoIn, int sectorStart, int sectorEnd) {
	if (!flowControlHook('P',"P prepare for write sectors %d to %d.",sectorStart,sectorEnd)) return false;
	return	lpcCommandUnsignedUnsigned(fd,fifoIn,'P',sectorStart,sectorEnd);
}

bool lpcErase(int fd, Fifo *fifoIn, int sectorStart, int sectorEnd) {
	if (!flowControlHook('E',"E erase sectors %d to %d.",sectorStart,sectorEnd)) return false;
	printfNormal("Erasing sectors %d to %d NOW\n",sectorStart,sectorEnd);
	return	lpcCommandUnsignedUnsigned(fd,fifoIn,'E',sectorStart,sectorEnd);
}

bool lpcUnlock(int fd, Fifo *fifoIn) {
	const char echo[] = { 'U', 0 };
	if (!flowControlHook('U',"U unlock.")) return false;
	return	fifoPrintString(&fifoOut,"U 23130")
		&& lpcSendAndCheck(fd,fifoIn,echo);
}

bool lpcCompare(int fd, Fifo *fifoIn, Uint32 addrA, Uint32 addrB, int n) {
	if (!flowControlHook('M',"M compare memory 0x%08X and 0x%08X, length=%d.",addrA,addrB,n)) return false;
	if (lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'M',addrA,addrB,n)) return true;
	else {
		printfDebug("Compare failed.\n");
		return false;
	}
}

/** lpcBlankCheck will not end in CMD_SUCCESS in case of non-blank sector(s). That's why we can't use lpcCommandUnsignedUnsigned.
 */
bool lpcBlankCheck(int fd, Fifo *fifoIn, int sectorStart, int sectorEnd, bool *blank) {
	const char echo[] = { 'I', 0 };

	if (!flowControlHook('I',"I blank check sectors %d to %d.",sectorStart,sectorEnd)) return false;
	if (fifoPrintString(&fifoOut,"I ")
	&& fifoPrintUDec(&fifoOut,sectorStart,1,10)
	&& fifoPrintChar(&fifoOut,' ')
	&& fifoPrintUDec(&fifoOut,sectorEnd,1,10)
	&& fifoPrintString(&fifoOut,"\r\n")
	&& lpcWriteFifo(fd,&fifoOut)) ;	// fine
	else {
		printfError("ERROR: output buffer overflow or IO problem.\n");
		return false;
	}
	Uint32 result;
	if (!lpcGetNonEchoAnswerLine(fd,fifoIn,echo)) return false;
/*
	if (fifoParseUnsigned(fifoIn,&result)) {
		printfDebug(" - return code %s\n",returnMessage(result));
		Uint32 offset,value;
		switch(result) {
		case 0:	*blank = true;
			if (fifoConsumeLine(fifoIn)) return true;
			else {
				printfError("ERROR: missing \\n\n");
				return true;
			}
		case 8:	// SECTOR_NOT_BLANK <offset> <value>
			*blank = false;
			if ((fifoParseBlanks(fifoIn), fifoParseUnsigned(fifoIn,&offset))
			&& (fifoParseBlanks(fifoIn), fifoParseUnsigned(fifoIn,&value))
			&& fifoConsumeLine(fifoIn) ) {
				printfDebug("Sector not blank at 0x%08X, value=0x%08X\n", offset,value);
				return true;
			}
			else {
				printfError("ERROR: could not read offset and value.\n");
				return false;
			}
		default:
			return false;
		}
	}
*/
	if (fifoFindUnsigned(fifoIn,&result)) {
		printfDebug(" - return code %s\n",returnMessage(result));
		Uint32 offset,value;
		switch(result) {
		case 0:	*blank = true;
			return true;
		case 8:	// SECTOR_NOT_BLANK \n <offset> \n <value>
			*blank = false;
			if (lpcFindUnsignedValues(fd,fifoIn,&offset,1)
			&&  lpcFindUnsignedValues(fd,fifoIn,&value,1)) {
				printfDebug("Sector not blank at sector[%d]+0x%08X, value=0x%08X\n",
					sectorStart,offset,value);
				return true;
			}
			else {
				printfError("ERROR: could not read offset and value.\n");
				return false;
			}
		default:
			printfError("ERROR: %s\n",returnMessage(result));
			return false;
		}
	}
	else {
		printfError("ERROR: could not read return code.\n");
		return false;
	}
}

bool lpcBailOut(int fd) {
	lpcWriteString(fd,"\x1B");
	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Read/Write functions
//
bool lpcReadUuencode(int fd, Fifo *fifoIn, Fifo *data, Uint32 address, Uint32 n) {
	const Uint32 r0 = fifoCanRead(data);

	if (lpcCommandUnsignedUnsigned(fd,fifoIn,'R',address,n)) {
		int checksum=0;
		for (int lineNo=0; fifoCanRead(data)-r0 <n; lineNo++) {
			if (fifoLoadLineOnDemand(fd,fifoIn) && fifoUuDecodeLine(data,fifoIn,&checksum)) ;	// fine
			else return lpcBailOut(fd);

			if ((lineNo % 20)==19 || fifoCanRead(data)-r0 ==n) {
				Uint32 checksumLpc;
				if (fifoLoadLineOnDemand(fd,fifoIn)
				&& fifoFindUnsigned(fifoIn,&checksumLpc)
				&& checksum==checksumLpc) {
					printfNormal(".");
					lpcWriteString(fd,"OK\r\n");	// :o) unchecked
				}
				else {
					printfError("ERROR: checksum invalid.\n");
					return lpcBailOut(fd);	// we do not retry
				}
				checksum = 0;
			}
		}
		return true;
	}
	else return false;
}

bool lpcReadBinary(int fd, Fifo *fifoIn, Fifo *data, Uint32 address, Uint32 n) {
	if (lpcCommandUnsignedUnsigned(fd,fifoIn,'R',address,n)) {
		if (fifoLoadN(fd,data,n)) {
			printfNormal(".");
			return true;
		}
		else return false;
	}
	else return false;
}

bool lpcRead(int fd, Fifo *fifoIn, int ispProtocol, Fifo *data, unsigned address, int n) {
	switch(ispProtocol) {
		case ISP_PROTOCOL_UUENCODE: return lpcReadUuencode(fd,fifoIn,data,address,n);
		case ISP_PROTOCOL_BINARY: return lpcReadBinary(fd,fifoIn,data,address,n);
		default: return lpcBailOut(fd);
	}
}

bool lpcWriteCommand(int fd, Fifo *fifoIn, Uint32 address, Uint32 size) {
	return lpcCommandUnsignedUnsigned(fd,fifoIn,'W',address,size);
}

/** Checked on LPC2136/01 -E and not -E.
 */
bool lpcWriteUuencode(int fd, Fifo *fifoIn, unsigned address, Fifo *data) {
	if (!flowControlHook('W',"W (write uuencode).")) return false;
	if (lpcWriteCommand(fd,fifoIn,address,fifoCanRead(data))) {
		int checksum=0;
		for (int lineNo=0; fifoCanRead(data); lineNo++) {
			if (!flowControlHook('w',"w (write uuencoded line).")) return false;
			if (fifoUuEncodeLine(&fifoOut,data,&checksum)) {
				//printfError("line %d\n",lineNo);
				lpcWriteFifo(fd,&fifoOut);
				if (useEcho) {
					if (!fifoLoadLineOnDemand(fd,fifoIn) || !fifoConsumeLine(fifoIn)) {
						printfError("ERROR: echo line missing.\n");
						return false;
					}
					printfDebug("\n");	// just formatting debug output
				}
			}
			else return lpcBailOut(fd);

			if ((lineNo%20)==19 || !fifoCanRead(data)) {	// checksum
				Uint32 checksumReturn;
				printfNormal(".");
				if (fifoPrintUDec(&fifoOut,checksum,1,10)
				&& fifoPrintString(&fifoOut,"\r\n")
				&& lpcWriteFifo(fd,&fifoOut)
				&& (!useEcho || lpcFindUnsignedValues(fd,fifoIn,&checksumReturn,1)
					&& checksum==checksumReturn)
				&& fifoLoadLineOnDemand(fd,fifoIn)
				&& fifoFindStringInLine(fifoIn,"OK")) {
					//fprintf(stderr,"Checksum confirmed\n");
				}
				else {
					printfError("ERROR: checksum invalid.\n");
					return lpcBailOut(fd);
				}

				checksum = 0;
			}
		}
		return true;
	}
	else return false;
}

bool lpcWriteBinary(int fd, Fifo *fifoIn, unsigned address, Fifo *data) {
	const int n = fifoCanRead(data);
	if (!flowControlHook('W',"W (write binary).")) return false;
	if (lpcWriteCommand(fd,fifoIn,address,fifoCanRead(data))) {

		// data per write command
		printfNormal("[%u=",n);

		// binary data is always echoed
		// Send data in small blocks and receive answer
		// This is a primitive kind of flow control.
		const Uint32 chunkSize = subBlockSize;	// 64 lead to a deadlock
		for (int o=0; o<n; o+= chunkSize) {
			const Uint32 cs = int32Min(chunkSize,n-o);

			Fifo fifoChunk;
			if (!fifoParseN(data,&fifoChunk,cs)) {
				printfError("ERROR: chunk extraction error.\n");
				return false;
			}

			if (!fifoPutFifo(&fifoOut,&fifoChunk)) {
				printfError("ERROR: output buffer overflow.\n");
				return false;
			}
			if (!flowControlHook('w',"w (write block[%d]).",(int)fifoCanRead(&fifoOut))) return false;
			if (!lpcWriteFifo(fd,&fifoOut)) {
				printfError("ERROR: IO error.\n");
				return false;
			}
			printfNormal("+%d",cs);

			if (!flowControlHook('r',"W (write block, read answer block[%d]).",(int)cs)) return false;
			if (useEcho) {
				if (fifoLoadN(fd,fifoIn,cs) && fifoCanRead(fifoIn)>=cs) {
					// we could compare here...
					fifoSkipRead(fifoIn,cs);
				}
				else {
					printfError("ERROR: binary echo mismatch.\n");
					printfError("Received data was: \"");
					printfErrorFifo(*fifoIn);
					printfError("\"\n");
					return false;
				}
			}
		}

/*
		if (!useEcho) {	// transfer without echo: \n is returned.
			if (fifoLoadLine(fd,fifoIn)
			&& fifoConsumeLine(fifoIn)) ;	// Maybe we should check more... :o)
			else {
				printfError("ERROR: \\n expected as binary response without echo.\n");
				return false;
			}
		}
*/
		printfNormal("]");

/*
		if (!fifoLoadLineOnDemand(fd,fifoIn)) {
			printfError("ERROR: no response after binary data.\n");
			return lpcBailOut(fd);
		}
		if (!fifoFindStringInLine(fifoIn,"OK")) {
			printfError("ERROR: missing response: \"OK\".");
			return lpcBailOut(fd);
		}
*/
		return true;
	}
	else return false;
}

bool lpcWrite(int fd, Fifo *fifoIn, int ispProtocol, unsigned address, Fifo *data) {
	switch(ispProtocol) {
		case ISP_PROTOCOL_UUENCODE: return lpcWriteUuencode(fd,fifoIn,address,data);
		case ISP_PROTOCOL_BINARY: return lpcWriteBinary(fd,fifoIn,address,data);
		default: return lpcBailOut(fd);
	}
}

bool lpcWriteN(int fd, Fifo *fifoIn, int ispProtocol, unsigned address, Fifo *data, int n) {
	Fifo fifoFraction;
	return	fifoParseN(data,&fifoFraction,n)
		&& lpcWrite(fd,fifoIn,ispProtocol,address,&fifoFraction);
}

typedef struct {
	bool 	verify;			///< verify flash after copy RAM to FLASH
	bool	fast;			///< waste FLASH lifetime for speed
	int	ramLowerSpace;		///< ISP uses 0x200 bytes at bottom of RAM, 0x270 on LPC800.
	int	ramUpperSpace;		///< ISP/IAP uses 32 + 0x100 bytes at top of RAM
} LpcProgramConfig;

/** Extended programming function for hex Images.
 *  @param fast erase sectors in one call, if not all blank. Otherwise erase sector for sector.
 * code read protection
 */
bool lpcProgramNew(int fd, Fifo *fifoIn, LpcFamilyMember const *member, const HexImage *hexImage, const LpcProgramConfig *conf) {
	const int blockLimit = member->sizeRamKs[0]*1024 - conf->ramLowerSpace - conf->ramUpperSpace;
	const LpcFamilyMemoryMap *map = member->familyMemoryMap;
	const int addressRam = map->addressRams[0] + conf->ramLowerSpace;
	printfNormal("Transfer RAM: %dkiB+%dB @ 0x%08X\n",blockLimit/1024,blockLimit%1024,addressRam);

	if (hexImageSegments(hexImage)!=1) {
		printfError("Cannot handle more than one segment so far, yet.\n");
		return false;
	}
	const int imageSize = hexImage->segments[0].size;
	const int address = hexImage->segments[0].address;

	// copy image to a Fifo
	char imageBuffer[512*1024];
	Fifo fifoImage = { imageBuffer, sizeof imageBuffer };
	for (int b=0; b<imageSize; b++) fifoWrite(&fifoImage,hexImage->segments[0].data[b]);

	// optimized for big images => maximum block with 'Copy RAM to FLASH' unless image is really small.
	int blockSize = 0;
	for (int i=0; i<ELEMENTS(map->blockSizes) && map->blockSizes[i]>0; ++i) {
		if (map->blockSizes[i] <= blockLimit
		&& blockSize<imageSize) 	blockSize = map->blockSizes[i];
	}
	// pad fifo for this block size
	while (fifoCanRead(&fifoImage) % blockSize) if (!fifoPrintChar(&fifoImage,0xFF)) return false;

	const int blocks = fifoCanRead(&fifoImage) / blockSize;
	const int blocksAtOnce = blockLimit / blockSize;	// the will be 1 at least

	printfNormal("Transmission RAM: %dblocks (%dkiB), units of %dB\n",
		blocksAtOnce, blocksAtOnce*blockSize/1024, blockSize);

	printfNormal("Flash address range 0x%08x..0x%08x, ",address,address+blockSize*blocks-1);
	int sectorFrom, sectorTo;
	if (lpcAddressRangeToSectorRange(member,&sectorFrom,&sectorTo,address,address+blockSize*blocks-1)) {
		printfNormal("sectors: %d..%d\n",sectorFrom,sectorTo);
	}
	else {
		printfError("ERROR: image does not fit into FLASH\n");
		return false;
	}

	if (!lpcUnlock(fd,fifoIn)) {
		printfError("ERROR: failed to unlock device.\n");
		return false;
	}

	// Manual of LPC800 states, that sector 0 cannot be blank checked (will always fail)
	// However, in reality, this seems to be wrong
	bool blank;
	if (!lpcBlankCheck(fd,fifoIn,sectorFrom,sectorTo,&blank)) {	// erasing required
		printfError("ERROR: failed to perform blank ckeck.\n");
		return false;
	}

	if (!blank) {	// erasing required
		if (conf->fast) {	// erase all sectors, even if not all of them are non-blank
			if (lpcPrepareForWrite(fd,fifoIn,sectorFrom,sectorTo)
			&& lpcErase(fd,fifoIn,sectorFrom,sectorTo)) printfNormal("Erased en bloc.\n");
			else {
				printfError("ERROR: failed to erase sectors %d to %d\n",sectorFrom,sectorTo);
				return false;
			}
		}
		else for (int s=sectorFrom; s<=sectorTo; ++s) {
			if (!lpcBlankCheck(fd,fifoIn,s,s,&blank)) {	// erasing required
				printfError("ERROR: failed to perform blank ckeck.\n");
				return false;
			}
			
			if (blank
			|| lpcPrepareForWrite(fd,fifoIn,s,s) && lpcErase(fd,fifoIn,s,s)) ;	// fine
			else {
				printfError("ERROR: failed to prepare/erase sector %d\n",s);
				return false;
			}
		}
	}

	for (int block=0; block<blocks; ) {
		// transfer bytes
		for (int subBlock=0; subBlock<blocksAtOnce && block+subBlock<blocks; ++subBlock) {
			printfNormal("\nTransfer block %d to RAM (%s) ",block+subBlock,
				member->ispProtocol==ISP_PROTOCOL_UUENCODE ? "uuencode" : "binary" );
			if (lpcWriteN(fd,fifoIn,member->ispProtocol,addressRam+subBlock*blockSize,&fifoImage,blockSize)) {	// fine
				printfNormal("OK.");
			}
			else {
				printfNormal("FAILED!\n");	// this is part of progress output
				printfError("ERROR: could not write block to RAM.\n");
				return false;
			}
		}
		// do flash programming
		for (int subBlock=0; subBlock<blocksAtOnce && block+subBlock<blocks; ++subBlock) {
			const Uint32 addressFlash = address+(block+subBlock)*blockSize;
			const int sector = lpcAddressToSector(member,addressFlash);
			printfNormal("\nTransfer block %d to FLASH, sector %d ",block+subBlock,sector);
			if (lpcPrepareForWrite(fd,fifoIn,sector,sector)
			&& lpcCopyRamToFlash(fd,fifoIn, addressFlash, addressRam+subBlock*blockSize, blockSize)); // fine
			else {
				printfError("ERROR: failed to Copy RAM to FLASH.\n");
				return false;
			}
			if (conf->verify) {
				if (addressFlash==0) {
					printfNormal(" verify not possible for sector 0.");
				}
				else {
					if (lpcCompare(fd,fifoIn,addressFlash,addressRam+subBlock*blockSize, blockSize)) 
					printfNormal(" verify OK.");
					else {
						printfNormal(" verify failed.\n");	// this is part of progress output
						printfError("ERROR: Verification failed at address 0x%08X.\n",addressFlash);
						return false;
					}
				}
			}
		}
		block += blocksAtOnce;
	}
	printfNormal("\n");

	return true;
}

static inline int min32(int a, int b) {
	return a<b ? a : b;
}

/** Re-writes a fifo, so that the LPCxxxx checksum (vector 7, offset 0x1C) is 0.
 */
bool fifoLpcChecksum(Fifo *fifo) {
	Fifo clone = *fifo;
	Fifo writer;
	fifoInitRewrite(&writer,fifo);

	Uint32 checksum = 0;
	for (int i=0; i<7; ++i) {
		Uint32 vector;
		if (!fifoGetUint32(&clone,&vector)) return false;
		checksum += vector;
	}
	fifoSkipWrite(&writer,4*7);
	return fifoPutInt32(&writer,-checksum);
}

// only correct for segment 0
bool lpcChecksumFix(const HexImage *hexImage) {
	Fifo fifo;
	fifoInitRead(&fifo, hexImage->segments[0].data, hexImage->segments[0].size);
	if (hexImage->segments[0].address==0) return fifoLpcChecksum(&fifo);
	else return true;
}

static unsigned address = 0;
static Int32 baud = 230400;
static Int32 crystalHz = 14748000;	// IRC + PLL of LPC17 and LPC23
static Int32 addressCrp = 0x2FC;	// code read protection word address
static const char * const defaultDevice = "/dev/ttyUSB0";
static Fifo fifoDevice = {};
static int eraseSymbol = -1;
static Int32 flashSizeOverride = -1;	// flash size override
static int manualSymbol = -1;
static Int32 levelCrpSet = 0;		// set this level of protection
static Int32 levelCrpMax = 0;		// limit the level of protection
static Int32 memorySizeOverride = -1;	// RAM 0 size override
static bool quick = false;		// prefer speed to flash lifetime
static Int32 readN = -1;
static bool showInfo = false;
static Int32 timeAfterResetMs = 200;
static Int32 timeoutSerialMs = 1000;	// 500 is enough for LPC17, but not the LPC21s
static bool verify = false;
static Int32 writeN = -1;		// write a specified number of bytes
static bool writeAll = false;		// write all
static Int32 executeAddress = -1;
static int executeSymbol = -1;
//static Int32 ucNumber = -1;		// force specific device
static int ucSymbol = -1;		// force user supplied.
static Fifo fifoF = {};
static Fifo fifoM = {};
static Fifo fifoX = {};			// execution control
static Fifo fifoN = {};			// controller name, provided by user
static Fifo fifoU = {};			// manually selected device
static const char *manuallySelected = 0;// user provides processor name
static Int32 ispProtocolOverride = -1;	// binary / uuencode override
static PairInt32 erase = { 0, -1 };
// since at least LPC800, bottom ISP RAM requirement has increased from 0x200 to 0x270
static Int32 ispReserved[2] = { 0x270, 0x120 };	// RAM used by ISP handler, bottom/top of RAM0
static Int32 userId = -1;
static Int32 listB[LPC_BLOCK_SIZES] = { };
static bool showAllDevices = false;	// command
static bool probeBaud = false;		// command
static bool showUid = false;		// command
static bool showBoot = false;		// command
static bool noUcOutput = false;		// output control: avoid default info of detected device.
static bool executeReset = false;	// command: start code by reset.

static const FifoPoptBool optionBools[] = {
	{ 'i',&showInfo },
	{ 'k',&showBoot },
	{ 'n',&noUcOutput },
	{ 'p',&probeBaud },
	{ 'q',&quick },
	{ 's',&showAllDevices },
	{ 'u',&showUid },
	{ 'v',&verify },
	{ 'w',&writeAll },
	{ 'x',&executeReset },
	{ 'E',&useEcho },
	{}
};

static const FifoPoptInt32Flag optionIntFlags[] = {
	{ 'z',&verbosity, VERBOSITY_SILENT },
	{ 'Z',&verbosity, VERBOSITY_SILENT_ERROR },
	{ 'g',&verbosity, VERBOSITY_DEBUG },
	{}
};

static const FifoPoptInt32 optionInts[] = {
	{ 'a',(Int32*)&address, &fifoParseIntEng },
	{ 'b',&baud, },
	{ 'c',&crystalHz, &fifoParseIntEng },
	{ 'C',&addressCrp, },
	{ 'f',&flashSizeOverride, &fifoParseIntEng },
	{ 'I',&userId, },
//	{ 'N',&userNumber, },
	{ 'l',&levelCrpSet, &fifoParseInt },
	{ 'L',&levelCrpMax, &fifoParseInt },
	{ 'm',&memorySizeOverride, &fifoParseIntEng },
	{ 'r',&readN, &fifoParseIntEng },
	{ 't',&timeAfterResetMs, },
	{ 'T',&timeoutSerialMs, },
//	{ 'U',&ucNumber, },
	{ 'j',&executeAddress, },
	{ 'y',&subBlockSize, },
	{}
};

static const char* eraseSymbols[] = {
	"all",
	0
};

static const char* executeSymbols[] = {
	"LMA","LMA+1",
	0
};

static const char* ucSymbols[] = {
	"user",
	0
};

static const char* manualSymbols[] = {
	"contents",
	"numeric",
	"new-devices",
	"crp",
	"flow-control",
	"questionable",
	"philosophy",
	0
};

static const char* ispProtocols[] = {
	"UUENCODE","BINARY",		// these positions must match the enum im lpcMemories.h !
	0
};

static const FifoPoptSymbol optionSymbols[] = {
	{ 'e',&eraseSymbol, eraseSymbols },
	{ 'h',&manualSymbol, manualSymbols },
	{ 'j',&executeSymbol, executeSymbols },
	{ 'U',&ucSymbol, ucSymbols },
	{ 'P',&ispProtocolOverride, ispProtocols },
	{}
};

static const FifoPoptString optionStrings[] = {
	{ 'd', &fifoDevice },
	{ 'F', &fifoF },
	{ 'M', &fifoM },
	{ 'X', &fifoX },	// execution flow control
	{ 'N', &fifoN },
	{ 'U', &fifoU },	// manual processor selection
	{}
};

static const FifoPoptInt32Range optionInt32Ranges[] = {
	{ 'e', &erase },
	{}
};

static const FifoPoptInt32Tuple optionInt32Tuples[] = {
	{ 'R', 2, ispReserved, ",", },
	{ 'B', 4, listB, ",", &fifoParseIntEng },
	{}
};

// complex parsing section: options -F and -M: lists of pairs

// end of option validator
bool eoo(Fifo *fifo) {
	return 0==fifoCanRead(fifo);
}

const FifoParsePairInt parseSizeXCount = {
	.parseIntA = &fifoParseIntEng,
	.separator = "x",
	.parseIntB = &fifoParseIntCStyle,
	// no validator!
};

const FifoParsePairInt parseSizeAtAddress = {
	.parseIntA = &fifoParseIntEng,
	.separator = "@",
	.parseIntB = &fifoParseIntCStyle,
	// no validator!
};

const FifoParseListPairInt parseF = {
	.parsePair = &parseSizeXCount,
	.separator = ",",
	.validator = &eoo
};

const FifoParseListPairInt parseM = {
	.parsePair = &parseSizeAtAddress,
	.separator = ",",
	.validator = &eoo
};

PairInt listF[3];
PairInt listM[3];

static void showHelp(void) {
	// small letters are mostly actions and basic settings.
	// capital letters are mostly options modifying default behaviour.
	printfOut(
	ISP " " ISP_VERSION_COPYRIGHT ". GPL license.\n"
	"A command-line ISP programmer for NXP LPC ARM devices.\n\n"
	"Usage: "ISP" <options> [<inputFile>]\n"
	"Options: (values in [] show the defaults, *=not yet implemented)\n"
	"  -? or --help              : display this help screen.\n"
	"  -0                        * use sector 0 even for questionable operations\n"
	"  -1                        * start with sector 1 instead of 0 for questionable operations\n"
	"  -a <destinationAddress>   : image destination address (LMA) (sector boundary!!) [0x%X]\n"
	"  -b <baud rate>            : communication baud rate [%d]\n"
	"  -c <f crystal/Hz>         : crystal frequency [%d]\n"
	"  -d <device>               : serial device [%s]\n"
	"  -e {<sect>[..<sect>]|all} : erase FLASH sectors from..to, or all\n"
	"  -f <flashSize>            : FLASH size override\n"
	"  -g                        : debugging output (in honor of gcc's option -g)\n"
	"  -h <topic>                : display manual <topic>, start with \"contents\"\n"
	"  -i                        : print member info\n"
	"  -j {addr|LMA|LMA+1}       : jump to address [unset]\n"
	"  -k                        : read boot code version (ISP 'K' command)\n"
	"  -l <CRP level>            : lock code. Set code read protection (CRP) level [%d]\n"
	"  -m <RAM size>             : RAM 0 size override\n"
	"  -n                        : do not output controller as default action\n"
	"  -o <output file>          : set output file (instead of standard output)\n"
	"  -p                        : probe baud rates and print to stdout. -q => only highest\n"
	"  -q                        : quick mode: prefer speed to security/kindness\n"
	"  -r <image size>           : read <image size> bytes from RAM/FLASH\n"
	"  -s                        : show all compiled-in devices.\n"
	"  -t <time/ms>              : time "ISP" waits after RESET deassertion [%d]\n"
	"  -u                        : output UID\n"
	"  -v                        : verify FLASH contents after programming\n"
	"  -w                        : write to uC (default, if <inputFile> provided\n"
	"  -x                        : execute by issuing RESET without BSL\n"
	"  -y <sub-block size>       : binary transfer sub-block size [%d]\n"
	"  -z                        : zero output in case of success\n"
//	"  -A                        :\n"
	"  -B 256,512,1024,4096      : define copy-ram-to-flash block sizes to be 256,512... \n"
	"  -C <CRP address>          : address of the 4-byte CRP code [0x%X]\n"
//	"  -D                        : debugging / delays / prompting.\n"
	"  -E                        : use echo on [normally off]\n"
	"  -F 4kix16,32kix14         : define FLASH sector layout to be 16x4kiB, then 14x32kiB\n"
//	"  -G                        :\n"
	"  -H                        * set error output handle [2][stderr]\n"
	"  -I <uC ID>                : set user-defined id\n"
//	"  -J                        :\n"
//	"  -K                        :\n"
	"  -L <CRP level>            : enable CRP level (dangerous especially for level 3) [%d]\n"
	"  -M 8ki@0x40000000,8ki@... : define uC RAM to be 8kiB at address 0x40000000 and ...\n"
	"  -N <uC number>            : set user-defined number\n"
//	"  -O                        :\n"
	"  -P {UUENCODE|BINARY}      : data encoding protocol override\n"
//	"  -Q                        :\n"
	"  -R <bottom>,<top>         : set RAM0 spaces reserved forISP[0x%03X,0x%03X]\n"
	"  -S [<from>,<to>]          * calculate CRC checksum starting at <from> until <to>\n"
	"  -T <time/ms>              : timeout of serial communication, step: 100ms [1000]\n"
	"  -U {<uC-number>|user}     : force uC like 2101, do not use part ID.\n"
	"  -V {CRC|READ}             * use verification method CRC or read & compare\n"
//	"  -W                        :\n"
	"  -X <flow-control>         : execution flow control, see -h flow-control\n"
//	"  -Y                        :\n"
	"  -Z                        : zero output, even in case of failure. (return code only)\n\n"
	"If your controller does not work correctly with isp, try low baud rate, low serial\n"
	"timeout (-T 1000 or more) and correct crystal frequency.\n"
	"Try -h contents for online manual\n"
	"\n"
	"Please report bugs to marc@windscooting.com\n"
	, address, baud, crystalHz, defaultDevice
	, levelCrpSet, timeAfterResetMs 
	, subBlockSize, addressCrp, levelCrpMax
	, ispReserved[0], ispReserved[1]
	, timeoutSerialMs

	);
}

void manualContents(void) {
	printfOut(
	"The following topics are available from the online manual:\n"
	"  -h philosophy             : motivation behind isp\n"
	"  -h numeric                : understanding isp's advanced command-line parsing\n"
	"  -h new-devices            : how to configure "ISP" for new devices\n"
	"  -h crp                    : code read protection (CRP)\n"
	"  -h flow-control           : debug output and control flow interruption\n"
	"  -h questionable           : operations consideres questionable\n"
	);
}

void manualQuestionable(void) {
	printfOut(
	"I (Marc, the author of this program) consider some operations of the LPC bootloader\n"
	"against common sense and in the context of this program these operations are called\n"
	"'questionable'. The questionable operations involve reading of sector 0. The manual(s)\n"
	"state that flash sector 0 cannot be read, because the boot ROM table is mapped into\n"
	"this location. So what? Isn't there a possibility to temporarily undo this mapping?\n"
	"However, it seems to me, that some devices implement common sense instead of user\n"
	"manual specifications, so I decided the user may select between options\n"
	"  -0 : include sector 0 in blank-check, verify\n"
	"  -1 : exclude sector 1 in blank-ckeck, verify\n"
	"For blank-check, assume the sector is NOT blank. For verify, assume the sector to\n"
	"be correct.\n"
	);
}

void manualNumericArguments(void) {
	printfOut(
	"In the author's opinion, graphic user interfaces may be good thing for beginners\n"
	"but are always a nuisance for the advanced (scripting) user. However, we haven't\n"
	"seen much progress in command-line interfaces for years. The author hopes, that\n"
	ISP "'s advanced command line features will inspire other programmers to make\n"
	"their (command line) programs more user-friendly. "ISP" uses single char options\n"
	"that accept arguments that are either structured (lists, tuples) or have\n"
	"numeric extensions. Integers can be decimal, 0xhexadecimal, sometimes accept\n"
	"suffixes like k (*1000) or ki (*1024) which are suitable to define common\n"
	"memory sizes. A few examples:\n"
	"  -c 14746k        means: 14,746,000 (Hz) crystal frequency.\n"
	"  -x 0x40000000    means: start program at 0x4000 0000, which is the same as\n"
	"  -x 1Gi           (1GiB above 0).\n"
	"  -e 1             erases sector 1 (the range 1 to 1 of sectors).\n"
	"  -e 1..3          erases sectors 1 to 3.\n"
	"  -e all           erases all sectors of the device in use.\n"
	"  -B 256,512,1024,4096  defines the possible transfer sizes used to copy RAM\n"
	"                   to flash for most (all?) LPCs. It is a 4..5-tuple.\n"
	"Defining the flash sector sizes is done with a list of pairs, with a bit of\n"
	"eye-candy added, as an example LPC213x:\n"
	"  -F 4kix8,32kix14,4kix5      as the short form of\n"
	"  -F 4096x8,32768x14,4096x5\n"
	"Please note the difference between 4k (4000) and 4ki (4096).\n"
	"Also, don't expect every integer to have all formats allowed.\n"
	);
}

void manualNewDevices(void) {
	printfOut(
	"If any of the options -F, -M, -B, -N is provided, then ALL of them must be provided.\n"
	"This defines a device. -I should be provided to enable automatic selection.\n"
	"Simply put: these option allow you to extend the compiled in table to your needs.\n"
	"In the author's opinion, you should not need more than a datasheet of a new device\n"
	"to program it. No recompilation required. "ISP" uses RAM of your controller, that's\n"
	"why you have to define RAM location and size, too.\n"
	"Option -U user forces use of user-defined layout as does -N xxxx -U xxxx.\n"
	"Normally "ISP" reads the controller's device ID (part ID) and searches for a matching\n"
	"entry from the command line or from the compiled-in table. You have the option to\n"
	"specify an explicit device number (-U 2102 for a LPC2102) to make a hard selection\n"
	"in cases where the part ID is not useful or not wanted. Specifying -U user will\n"
	"always select your memory specification on the command line. Two convenience\n"
	"options are provided to specify total FLASH size (-f) and RAM0 size (-m) while still\n"
	"drawing most information out of the compiled-in table.\n"
	"Example: you want to program LPC2102. This controller has the same ID as LPC2103,\n"
	"yet the memory sizes are different.\n"
	"Option 1: define LPC2102 from scratch:\n"
	"  -F 4kix4 -M 4ki@1Gi -B 256,512,1024,4096 -N 2102 -U user\n"
	"Option 2: use compiled-in table, which yields LPC2103 and limit the memories:\n"
	"  -f 16ki -m 4ki\n"
	"Option 3: use compiled-in table, specify device number:\n"
	"  -U 2102\n"
	);
}

void manualCrp(void) {
	printfOut(
	ISP " takes care of CRP in the following sense: avoid unintended CRP and add CRP\n"
	"where needed. Achieving CRP3 is intentionally a bit hard to achieve.\n"
	ISP " allows upgrading and downgrading of CRP levels 0..2.\n"
	"CRP levels are encoded at memory location 0x2FC (4 bytes). The following rules apply:\n"
	"  a.CRP3 will only be executed if -l 3, -L 3 are given and the image contains\n"
	"    the pattern 0x43218765 at 0x2FC.\n"
	"  b.If 0x2FC contains a CRP pattern and CRP is not enabled (-L) on the command\n"
	"    line an error is issued.\n"
	"  c.If 0x2FC does not contain -1 or 0x12345678 CRP levels 1,2 cannot be applied.\n"
	"Rule a. inhibits unintentional CRP3 if the program does not provide IAP.\n"
	"Rule b. inhibits accidental CRP by the compiler/linker.\n"
	"Rule c. inhibits unwanted executable modification by isp.\n"
	"To unlock a CRP-enabled image use -l 0.\n"
	);
}

void manualFlowControl(void) {
	printfOut(
	"Using isp's flow control mechanism, the user can insert delays or confirmation\n"
	"requests into the normal program flow. Rudimentary knowledge of the ISP protocol\n"
	"is required for this. Program flow can be interrupted/delays before every ISP command\n"
	"and in case of data transfer (ISP block sizes) before isp's sub-block-transfers.\n"
	"If ISP mode 'echo on' is used, any data write is followed by a data read, to which\n"
	"the 'cmd' r applies.\n"
	"  -X {<cmd>[<action>]|.}+   : execution flow control / prompt\n"
	"    where <cmd> may be one of the following (mostly) ISP command letters\n"
	"      '1'  : write of a single byte\n"
	"      'A'  : change echo\n"
	"      'B'  : change baud rate\n"
	"      'C'  : copy RAM to FLASH\n"
	"      'E'  : erase sector(s)\n"
	"      'G'  : go (execute code)\n"
	"      'I'  : blank check sectors\n"
	"      'J'  : read part ID\n"
	"      'K'  : read boot code version\n"
	"      'M'  : compare memory\n"
	"      'N'  : read part UID\n"
	"      'P'  : prepare sector(s) for writing\n"
	"      'R'  : read memory\n"
	"      'r'  : data block read memory\n"
	"      'S'  : read CRC checksum\n"
	"      'U'  : unlock\n"
	"      'W'  : write to RAM\n"
	"      'w'  : data block of write to RAM\n"
	"    <action> can be:\n"
	"      omitted   : prompt user before proceeding. Program is stopped.\n"
	"      y         : prompt user, default to yes (continue)\n"
	"      n         : prompt user, default to no (quit program)\n"
	"      d<time>   : delay program by <time>\n"
	"    <time> can be:\n"
	"      <n>s      : <n> seconds.\n"
	"      <n>ms     : <n> milliseconds (may not be accurate).\n"
	"      <n>us     : <n> microseconds (may not be accurate).\n"
	"      <n>c      : <n> current baud rate charater timings.\n"
	"    The user can react to the prompt by:\n"
	"      - simply hitting enter. This takes the default action.\n"
	"      - typing y or Y. This continues the program.\n"
	"      - typing n or N. This terminates the program.\n"
	"    The character '.' can be used for improvind readability - it has no effect.\n"
	);
}

void manualPhilosophy(void) {
	printfOut(
	"The program "ISP" was originally designed because of the lack of well-behaved (*) ISP programmers\n"
	"for the NXP LPC21xx family.\n"
	"\n"
	"Well-behaved means: easily controllable by scripts/makefiles.\n"
	"\n"
	"Well-behaved means: reliability of operation (no blocking/deadlocking/need for killing it).\n"
	ISP " terminates after short time. Always. It does repond to SIGINT, although you don't need that.\n"
	"\n"
	"Well-behaved means: provide options to influence the program but don't require the user to use\n"
	"these options on every invokation. Which leads us directly to the next point:\n"
	"\n"
	"Well-behaved means: simple invokation and reasonable defaults for the options.\n"
	"\n"
	"Well-behaved means: provide functionality, not policy.\n"
	"You can use "ISP" for retrieving information from the microcontroller and use that information in\n"
	"a script. For example you can use "ISP" to read the UID and output the UID (and only the UID and\n"
	"and nothing but the UID) to stdout. You can write a certain sector of the controller with some\n"
	"data important to you - it does not have to be a program at all.\n"
	"\n"
	"Well-behaved means: it is NOT a GUI program, asshole, if you didn't get it until here!! If you\n"
	"plan to provide a user interface to this program - a pretty simple task, because of isp's\n"
	"script-oriented design (funny, isn't it?) - please DON'T DO it. Mentally disabled persons should\n"
	"not do programming at all. Don't let them multiply their bullshit. Please move on and reduce the\n"
	"functionality of another program, thank you very much.\n"
	"\n\n"
	"(*) from the author's point of view.\n"
	);
}

bool fifoLpcCrp(Fifo *image, Uint32 addressCrp, Uint32 address, int crpMax, int crp) {

	if (addressCrp&3 || address&3) {
		printfError("Alignment error of destination address or CRP bytes.\n");
		return false;
	}
	if (crpMax>3 || crpMax<0) {
		printfError("Maximum CRP level (-L) out of range.\n");
		return false;
	}
	if (crp>3 || crp<0) {
		printfError("CRP level (-l) out of range.\n");
		return false;
	}
	if (crp>crpMax) {
		printfError("CRP level (-l) not allowed. Allow with option -L\n");
		return false;
	}
	if (addressCrp<address) {
		if (crp>0) {
			printfError("ERROR: cannot provide CRP: destination address too high.\n");
			return false;
		}
		else return true; 	// nothing to do
	}

	Fifo clone = *image;
	if (fifoCanRead(&clone)>addressCrp-address) {
		fifoSkipRead(&clone,addressCrp-address);
		Fifo writer;
		fifoInitRewrite(&writer,&clone);
		Uint32 crpWord = 0;
		if (!fifoGetUint32(&clone,&crpWord)) return false;
		const int crpExe = lpcCrpCodeToLevel(crpWord);
		if (crpExe>crpMax) {
			printfWarning("WARNING: image has CRP (level %d)\n",crpExe);
		}
		if ((crp==3 || crpMax==3 || crpExe==3)		// attempt for CRP3
		&& !(crp==3 && crpMax==3 && crpExe==3)) {	// requirement for CRP3 
			printfError("ERROR: CRP level 3 needs -l3 -L3 and image to contain 0x43218765@0x2FC.\n");
			return false;
		}
		if (crp==-1 && crpExe>crpMax) {
			printfError("ERROR: image CRP (level %d) too high\n",crpExe);
			return false;
		}
		// setting crp level
		if (crp!=-1 && crpExe!=crp) {
			if (crpWord==~0u	// erased memory pattern
			|| crpExe>0) {		// any CRP level
				printfWarning("WARNING: image location 0x%04X (CRP) modified.\n",addressCrp);
				fifoPutInt32(&writer,lpcCrpLevelToCode(crp));
			}
			else {
				printfError("ERROR: image location 0x%04X (CRP) must be blank (-1)"
						" or CRP level 1..3\n",addressCrp);
				return false;
			}
		}
		return true;
	}
	else {
		if (crp==0) return true;	// we don't try, what we can't
		else {
			printfError("ERROR: image too small for CRP.\n");
			return false;
		}
	}
}

bool lpcCrp(const HexImage *hexImage, Uint32 addressCrp, int crpMax, int crp) {
	Fifo fifo;
	fifoInitRead(&fifo,(char*)hexImage->segments[0].data, hexImage->segments[0].size);
	return fifoLpcCrp(&fifo,addressCrp,hexImage->segments[0].address,crpMax,crp);
}

int noConnection(void) {
	printfError("ERROR: no connection - try different baud rate and especially crystal frequency?\n");
	return 1;
}

bool trySerialParams(const char *device, Uint32 baud, Uint32 timeAfterResetMs, Uint32 timeoutSerialMs) {
		printfDebug("Probing: "ISP" -d%s, -b%d -t%d -T%d\n",device,baud,timeAfterResetMs,timeoutSerialMs);
		int fd = serialOpenBlockingTimeout(device,baud,timeoutSerialMs/100);

		lpcReset(fd,true);
		lpcBsl(fd,true);
		usleep(10*1000);
		lpcReset(fd,false);
		usleep(timeAfterResetMs*1000);	// DS1818 resets for at least 150ms
		lpcBsl(fd,false);

		const bool result = lpcSync(fd,12*1000*1000);
		close(fd);
		return result;
}

int probeBauds(const char *device, Uint32 timeAfterReset, Uint32 timeoutSerialMs, bool quick) {

	int highest = 0;
	Uint32 bauds[] = { 230400, 115200, 57600, 38400, 19200, 9600 };
	for (int i=0; i<ELEMENTS(bauds); ++i) {
		if (trySerialParams(device,bauds[i],timeAfterReset,timeoutSerialMs)) {
			highest = int32Max(highest,bauds[i]);
			printfOut(ISP " -d%s, -b%d -t%d -T%d\n",device,bauds[i],timeAfterResetMs,timeoutSerialMs);
			if (quick) break;
		}
	}
	return highest;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Flow control hook
enum FlowControlAction {
	FC_NO_ACTION,		///< no action
	FC_DELAY,		///< unconditional delay
	FC_WAIT_YES,		///< wait for user input
	FC_WAIT_NO,		///< wait for user input
};

typedef struct {
	int		command;	///< most often, the ISP command letter.
	int		action;
	int		delayUs;	///< negative values indicate character units
} FlowControl;

static FlowControl flowControls[] = {
	{ '1' },
	{ 'A' },
	{ 'B' },
	{ 'C' },
	{ 'E' },
	{ 'G' },
	{ 'I' },
	{ 'J' },
	{ 'K' },
	{ 'M' },
	{ 'N' },
	{ 'P' },
	{ 'R' },
	{ 'S' },
	{ 's' },
	{ 'U' },
	{ 'W' },
	{ 'w' },
};

Int64 baudBitsToUs(int baudBits) {
	return 1000ll*1000*baudBits/baud;
}

FlowControl* flowControlFind(int command) {
	for (int i=0; i<ELEMENTS(flowControls); i++)
		if (command==flowControls[i].command) return &flowControls[i];
	return 0;
}

bool flowControlConfig(int command, int action, int delayUs) {
	FlowControl *fc = flowControlFind(command);
	if (fc!=0) {
		fc->action = action;
		fc->delayUs = delayUs;
		return true;
	}
	else return false;
}

static void safeUsleep(long long us) {
	if (us<1000*1000) usleep((int)us);
	else {
		struct timespec ts = {
			.tv_sec = us / 1000000,
			.tv_nsec = (us % 1000000) * 1000
		};
		nanosleep(&ts,0);
	}
}

bool flowControlHook(char command, const char* format, ...) {
	FlowControl *fc = flowControlFind(command);
	if (fc==0 || fc->action==FC_NO_ACTION) return true;
	else {
		switch(fc->action) {
			case FC_DELAY:
				if (fc->delayUs>=0) safeUsleep(fc->delayUs);
				else safeUsleep(baudBitsToUs(-fc->delayUs*10/*bits per char*/));
				return true;
			case FC_WAIT_YES:
			case FC_WAIT_NO: {
				const bool yes = fc->action==FC_WAIT_YES;
				printfNormal("Stopped at: ");
				va_list args;
				va_start(args,format);
				vprintfNormal(format,args);
				va_end(args);
				printfNormal(" - Continue %s", yes ? "([Y]/n) ? " : "(y/[N]) ? ");
				char line[100];
				const int n = read(0,line,sizeof line -1);
				if (n>=1) {
					line[n] = 0;
				}
				else {
					printfError("ERROR: could not read user input.\n");
					return false;
				}
				switch(line[0]) {
					case 0:
					case '\n':
					case '\r':	if (!yes) printfNormal("Operation canceled by user.\n");
							return yes;	// the default
					case 'y':
					case 'Y':	return true;
					case 'n':
					case 'N':	printfNormal("Operation canceled by user.\n");
							return false;
					default :	printfError("ERROR: invalid user input.\n");
							return false;
				}
			}
			default: 
				printfError("ERROR: invalid control flow action.\n");
				return false;
		}
	}
}

bool startsWith(const char *prefix, const char *string) {
	while (*prefix) {
		if (*prefix!=*string) return false;
		prefix++; string++;
	}
	return true;
}

int main(int argc, const char **argv) {
	struct timespec ts;
	clock_getres(CLOCK_REALTIME,&ts);
	printfDebug("Clock resolution: %lds %ldns\n",ts.tv_sec,ts.tv_nsec);

	char lineBuffer[1024];
	Fifo fifoCmdLine = { lineBuffer, sizeof lineBuffer };
	Fifo fifoNonOptions = { lineBuffer, sizeof lineBuffer };

	if (!fifoPoptAccumulateCommandLine(&fifoCmdLine,argc,argv)) {
		printfError("ERROR: command line too long.\n");
		return 1;
	}

	if (fifoPoptScanOption(&fifoCmdLine,'?',"help")) {
		showHelp();
		return 0;
	}

	while (fifoCanRead(&fifoCmdLine)) {
		if (false
		|| fifoPoptBool(&fifoCmdLine,optionBools)
		|| fifoPoptInt32Flag(&fifoCmdLine,optionIntFlags)
		|| fifoPoptInt32(&fifoCmdLine,optionInts)
		|| fifoPoptInt32Range(&fifoCmdLine,optionInt32Ranges)
		|| fifoPoptInt32Tuple(&fifoCmdLine,optionInt32Tuples)
		|| fifoPoptString(&fifoCmdLine,optionStrings)
		|| fifoPoptSymbol(&fifoCmdLine,optionSymbols)
		|| fifoPoptNonOptionAccumulate(&fifoCmdLine,&fifoNonOptions)
		) {
			// fine
		}
		else {
			printfError("ERROR: invalid option/parameter: \"%s\"\n",fifoReadPositionToString(&fifoCmdLine));
			printfError("Try \"-?\" for help\nTry \"-h contents\" for manual overview\n");
			return 1;
		}
	}

	if (manualSymbol>-1) {
		switch(manualSymbol) {
			case 0: manualContents(); return 0;
			case 1: manualNumericArguments(); return 0;
			case 2: manualNewDevices(); return 0;
			case 3: manualCrp(); return 0;
			case 4: manualFlowControl(); return 0;
			case 5: manualQuestionable(); return 0;
			case 6: manualPhilosophy(); return 0;
			default: printfError("ERROR: internal manual error\n"); return 1;
		}
	}
	// early parsing
	int nFs = 0;
	int nMs = 0;
	if (fifoCanRead(&fifoM)) {
		if (1<=(nMs=fifoParseListPairInt(&fifoM,&parseM,listM,ELEMENTS(listM)))) {
			printfDebug("Explicit memory definition: %dkiB RAM at 0x%08X.\n",listM[0].a/1024,listM[0].b);
		}
		else {
			printfError("ERROR: invalid explicit memory definition.\n");
			return 1;
		}
	}
	if (fifoCanRead(&fifoF)) {
		if (1<=(nFs=fifoParseListPairInt(&fifoF,&parseF,listF,ELEMENTS(listF)))) {
			printfDebug("Explicit sector layout definition:");
			for (int i=0; i<nFs; ++i) {
				if (i!=0) fprintf(stderr,", ");
				printfDebug("%d sectors of %dkiB",listF[i].b,listF[i].a/1024);
			}
			printfDebug("\n");
		}
		else {
			printfError("ERROR: invalid explicit sector layout definition.\n");
			return 1;
		}
	}
	while (fifoCanRead(&fifoX)) {
		const char cmd = fifoRead(&fifoX);
		if (cmd=='.') continue;	// . as optional separator
		FlowControl *fc = flowControlFind(cmd);
		if (fc!=0) {	// valid command
			if (fifoParseExactChar(&fifoX,'y')) {
				if (!flowControlConfig(cmd,FC_WAIT_YES,0)) return 1;
			}
			else if (fifoParseExactChar(&fifoX,'n')) {
				if (!flowControlConfig(cmd,FC_WAIT_NO,0)) return 1;
			}
			else if (fifoParseExactChar(&fifoX,'d')) {
				int n;
				int power;
				if (fifoParseInt(&fifoX,&n)) {
					if (fifoParseTechnicalUnit(&fifoX,&power,"s")) {
						switch(power) {
						case 0: if (!flowControlConfig(cmd,FC_DELAY,n*1000*1000)) return 1;
							break;
						case -3: if (!flowControlConfig(cmd,FC_DELAY,n*1000)) return 1;
							break;
						case -6: if (!flowControlConfig(cmd,FC_DELAY,n)) return 1;
							break;
						default:	printfError("ERROR: value out of range/resolution.\n");
								return 1;
						}
					}
					else if (fifoParseTechnicalUnit(&fifoX,&power,"c")) {
						switch(power) {
						case 0: if (!flowControlConfig(cmd,FC_DELAY,-n)) return 1;
							break;
						case 3: if (!flowControlConfig(cmd,FC_DELAY,-n*1000)) return 1;
							break;
						case 6: if (!flowControlConfig(cmd,FC_DELAY,-n*1000*1000)) return 1;
							break;
						default:	printfError("ERROR: value out of range/resolution.\n");
								return 1;
						}
					}
					else {
						printfError("ERROR: invalid unit.\n");
						return 1;
					}
				}
				else {
					printfError("ERROR: delay expected.\n");
				}
			}
		}
		else {
			printfError("ERROR: flow control on command '%c' not implemented.\n",cmd);
			return 1;
		}
	}
	// check, if execution specification is not contradictory:
	if (executeReset) {	// RESET method specified
		if (executeSymbol!=-1 || executeAddress!=-1) {
			printfError("ERROR: specifying both -x and -j is contradictory.\n");
			return 1;
		}
	}

	// Explicit file on command line is intended for writing as default.
	const char *inputFile = fifoCanRead(&fifoNonOptions) ? fifoReadPositionToString(&fifoNonOptions) : 0;
	if (inputFile) writeAll = true;

	const char *device = fifoCanRead(&fifoDevice) ? fifoReadPositionToString(&fifoDevice) : defaultDevice;
	
	const LpcFamilyMember * const * const lpcFamilyMembers = lpcFamilyMembersXxxx;	// used in this program

	// First command: list compiled-in devices
	if (showAllDevices) {
		printfOut("Compiled-in devices:\n");
		for (int d=0; lpcFamilyMembers[d]!=0 ; d++) {
			const LpcFamilyMember *m = lpcFamilyMembers[d];
			printfOut("  ID 0x%08X : %s\n",m->id,m->name);
		}
	}

	// Command:
	if (probeBaud) {
		baud = probeBauds(device,timeAfterResetMs,timeoutSerialMs,quick);
		if (baud==0) return 1;
	}

	printfDebug("Communication device=%s, speed=%dbd, uC crystal=%dkHz\n",device,baud,crystalHz/1000);
	int fd = serialOpenBlockingTimeout(device,baud,timeoutSerialMs/100);

	lpcReset(fd,true);
	lpcBsl(fd,true);
	usleep(10*1000);
	lpcReset(fd,false);
	usleep(timeAfterResetMs*1000);	// DS1818 resets for at least 150ms
	lpcBsl(fd,false);

	Uint32 partId = 0;
	if (lpcSync(fd,crystalHz)
	&& lpcBaud(fd,baud,1)
	&& lpcEcho(fd,&fifoIn,useEcho)
	&& lpcReadPartId(fd,&fifoIn,&partId)
	) ;	// fine
	else return noConnection();

	// calculate device
	// user (command-line) defined device...
	LpcFamilyMemoryMap familyMemoryMapUser = {};
	LpcFamilyMember familyMemberUser = {
		.familyMemoryMap = &familyMemoryMapUser
	};
	//bool userDefinedAvailable = false;

	if (nFs>=1		// option -F
	|| nMs>=1		// option -M
	|| listB[0]!=0		// option -B
	|| fifoCanRead(&fifoN)	// option -N
	|| userId!=-1		// option -I
	) if (nFs>=1 && nMs>=1 && listB[0]!=0 && fifoCanRead(&fifoN)) {
		if (nFs<1) {
			printfError("Invalid FLASH layout definition.\n");
			return 1;
		}
		familyMemoryMapUser.addressFlash = 0;
		for (int ss=0; ss<nFs; ++ss) {
			familyMemoryMapUser.sectorArrays[ss].sizeK = listF[ss].a / 1024;
			familyMemoryMapUser.sectorArrays[ss].n = listF[ss].b;
		}

		if (nMs<1) {
			printfError("Invalid RAM layout definition.\n");
			return 1;
		}
		for (int ms=0; ms<nMs; ++ms) {
			familyMemberUser.sizeRamKs[ms] = listM[ms].a / 1024;
			familyMemoryMapUser.addressRams[ms] = listM[ms].b;
		}

		for (int b=0; b<ELEMENTS(familyMemoryMapUser.blockSizes); ++b)
			familyMemoryMapUser.blockSizes[b] = listB[b];

		familyMemberUser.sizeFlashK = lpcFamilySectorOffset(&familyMemoryMapUser,
			lpcFamilySectorHighest(&familyMemoryMapUser)+1) / 1024;
		familyMemberUser.name = fifoCanRead(&fifoN) ? fifoReadLinear(&fifoN) : "LPC 0";
		familyMemberUser.id = userId;		// -1 hopefully matches no real device. 0 neither.
		//userDefinedAvailable = true;
	}
	else {
		printfError("All of the options -F, -M, -B, -N are required for defining new device.\n");
		return 1;
	}

	// select device: user/probed by id/number
	if (fifoCanRead(&fifoU)) manuallySelected = fifoReadLinear(&fifoU);	// uC selected by -U

	// the user-defined device is always checked first, to be able to override table entries
	LpcFamilyMember member = {};
	LpcFamilyMemoryMap memoryMap = {};
	bool deviceDetected = true;
	if (ucSymbol!=-1 && manuallySelected!=0) {
		printfError("Competing options -U.\n");
		return 1;
	}
	
	if (ucSymbol==0) {			// 'user' selected, the user's command line-defined device.
		member = familyMemberUser;
		memoryMap = familyMemoryMapUser;
		deviceDetected = false;
	}
	else if (manuallySelected) {
		LpcFamilyMember const *tMember = lpcFindByName(lpcFamilyMembers,manuallySelected);
		if (tMember!=0) {
			member = *tMember;
			memoryMap = *tMember->familyMemoryMap;
			deviceDetected = false;
		}
		else {
			printfError("Cannot find %s\n",manuallySelected);
			return 1;
		}
	}
	else {	// find by ID
		if (partId==familyMemberUser.id) {
			member = familyMemberUser;
			memoryMap = familyMemoryMapUser;
		}
		else {
			LpcFamilyMember const *tMember = lpcFindById(lpcFamilyMembers,partId);
			if (tMember!=0) {
				member = *tMember;
				memoryMap = *tMember->familyMemoryMap;
			}
			else {
				printfError("Cannot find part ID0x%08X",partId);
				return 1;
			}
		}
	}
	member.familyMemoryMap = &memoryMap;	// pointer rearrangement required.
	printfDebug("%s: %s, FLASH:%dkiB, RAM0:%dkiB, RAM1:%dkiB, RAM2:%dkiB\n",
		deviceDetected ? "Detected" : "Selected",
		member.name,member.sizeFlashK,member.sizeRamKs[0],member.sizeRamKs[1],member.sizeRamKs[2]
	);

	if (!noUcOutput && deviceDetected) {
		printfNormal("Detected: %s, FLASH:%dkiB, RAM0:%dkiB, RAM1:%dkiB, RAM2:%dkiB\n",
			member.name,member.sizeFlashK,member.sizeRamKs[0],member.sizeRamKs[1],member.sizeRamKs[2]
		);
	}

	// now patch the device with command line overrides
	if (flashSizeOverride!=-1) member.sizeFlashK = flashSizeOverride/1024;
	if (memorySizeOverride!=-1) member.sizeRamKs[0] = memorySizeOverride/1024;
	if (ispProtocolOverride!=-1) member.ispProtocol = ispProtocolOverride;

	// Command: show info, if desired
	if (showInfo) {
		char infoBuffer[2048];
		Fifo fifoInfo = { infoBuffer, sizeof infoBuffer };
		fifoPrintLpcFamilyMember(&fifoInfo,&member);
		printfOut("Device information:\n");
		printFifo(stdout,&fifoInfo);
		printfOut("\n");
	}

	if (showBoot) {
		Uint32 bootCodeVersion = 0;
		if (lpcReadBootCodeVersion(fd,&fifoIn,&bootCodeVersion)) {
			printfOut("%d.%d\n",bootCodeVersion>>8,bootCodeVersion&0xff);
		}
		else return 1;
	}

	if (showUid) {
		Uint32 uid[4];
		if (lpcReadUid(fd,&fifoIn,uid)) {
			printfOut("0x%08X 0x%08X 0x%08X 0x%08X\n",uid[0],uid[1],uid[2],uid[3]);
		}
		else {
			printfError("ERROR: cannot read UID.\n");
			return 1;
		}
	}

	// Command: read image if desired
	if (-1!=readN) {
		char imageBuffer[512*1024];
		Fifo fifoImage = { imageBuffer, sizeof imageBuffer };

		printfNormal("Read FLASH [%d] ",readN);
		if (lpcRead(fd,&fifoIn,member.ispProtocol,&fifoImage,address,readN)) {
			if (ioWriteFifo(1,&fifoImage)) {	// write to stdout, fd 1
				printfNormal(" OK.\n");
			}
			else {
				printfError("ERROR: Broken pipe.\n");
				return 1;
			}
		}
		else {
			printfError("ERROR: cannot read FLASH.\n");
			return 1;
		}
	}

	// explicit erasure of sectors
	const bool eraseAll = eraseSymbol==0;
	if (eraseAll) {
		erase.a = 0;
		erase.b = lpcAddressToSector(&member, member.familyMemoryMap->addressFlash+member.sizeFlashK*1024 -1);
		if (erase.b==-1) {
			printfError("Failed to calculate last sector\n");
			return 1;
		}
	}

	if (erase.a<=erase.b) {
		if (true
		&& lpcUnlock(fd,&fifoIn)
		&& lpcPrepareForWrite(fd,&fifoIn,erase.a,erase.b)
		&& lpcErase(fd,&fifoIn,erase.a,erase.b)) {
			fprintf(stderr,"Erased sectors %d to %d\n",erase.a,erase.b);
		}
		else {
			fprintf(stderr,"Failed to erase sectors %d to %d\n",erase.a,erase.b);
			fflush(stderr);
			return 1;
		}
	}

	// writing
	if (writeAll || writeN!=-1) {
		if (writeAll && writeN!=-1) {
			printfError("Both write N and write all requested.\n");
			return 1;
		}

		// writeN only limits the maximum size; does not force padding in the first place
		/*
		const int inputFd = inputFile!=0 ? open(inputFile,O_RDONLY) : 0;
		if (inputFd==-1) {
			printfError("Cannot open input file \"%s\"\n",inputFile);
			return 1;
		}
		char c;
		for (int b=0; (writeN==-1 || b<writeN) && (fifoCanWrite(&fifoImage) && 1==read(inputFd,&c,1)); b++) {
			fifoWrite(&fifoImage,c);
		}
		close(inputFd);
		*/

		Uint8 hexImageBuffer[512*1024];
		HexImage hexImage = { hexImageBuffer, sizeof hexImageBuffer, };
		hexImageInit(&hexImage);
		if (!hexFileLoad(inputFile,&hexImage)) {
			printfError("ERROR: could not load file \"%s\".\n",inputFile ? inputFile : "<stdin>");
			return 1;
		}
		if (hexImageSegments(&hexImage)!=1) {
			printfError("ERROR: %d file segments in image, but only 1 supported.\n",
				hexImageSegments(&hexImage));
			return 1;
		}
		if (address!=0) hexImage.segments[0].address = address;	// manual override

		// handle checksum in image vector table entry #7
		if (lpcChecksumFix(&hexImage)) {
			const int n = hexImage.segments[0].size;
			printfNormal("Image: %d = 0x%06X Bytes\n", n,n);
		}
		else {
			printfError("ERROR: image too small for (required) checksum calculation.\n");
			return 1;
		}

		// handle code locking
		//if (!fifoLpcCrp(&fifoImage,addressCrp,address,levelCrpMax,levelCrpSet)) {
		if (!lpcCrp(&hexImage,addressCrp,levelCrpMax,levelCrpSet)) {
			printfError("ERROR: CRP problem.\n");
			return 1;
		}

		const LpcProgramConfig lpcProgramConfig = {
			.verify = verify,
			.fast = quick,
			.ramLowerSpace = ispReserved[0],
			.ramUpperSpace = ispReserved[1]
		};
		if (lpcUnlock(fd,&fifoIn) && lpcProgramNew(fd,&fifoIn,&member,&hexImage,&lpcProgramConfig)) ;	// fine
		else {
			printfError("ERROR: cannot program FLASH.\n");
			return 1;
		}
	}

	// program execution
	if (executeReset) {
		flowControlHook('G',"Start program by system reset, NOT ISP command GO.");
		lpcReset(fd,true);
		lpcBsl(fd,false);
		usleep(timeAfterResetMs*1000);	// DS1818 resets for at least 150ms
		lpcReset(fd,false);
		printfNormal("Started program by system RESET\n");
	}
	else {
		if (executeSymbol!=-1) {
			switch(executeSymbol) {
				case 0:	// LMA = ARM-mode at load address
					executeAddress = address;
					break;
				case 1:	// LMA+1 = THUMB-mode at load address
					executeAddress = address+1;
					break;
				default:
					printfError("ERROR: invalid executeSymbol.\n");
					return 1;
			}

		}
		if (executeAddress!=-1) {
			if (lpcUnlock(fd,&fifoIn) && lpcGo(fd,&fifoIn,executeAddress&~1,executeAddress&1)) {
				// no return from here!
				printfNormal("Started program at address 0x%08X\n",executeAddress);
			}
			else {
				printfError("ERROR: failed to execute.\n");
				return 1;
			}
		}
	}
	// restore settings
	//tcsetattr(fd,TCSANOW,&terminalSettingsSaved);

	close(fd);

	return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

int printfDebug(const char *format, ...) {
	if (verbosity>=VERBOSITY_DEBUG) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on stderr.
 */
int printfNormal(const char *format, ...) {
	if (verbosity>=VERBOSITY_NORMAL) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

 
/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on stderr.
 */
int vprintfNormal(const char *format, va_list args) {
	if (verbosity>=VERBOSITY_NORMAL) {
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		return n;
	}
	return 0;
}

 
/** Prints out error messages normally expected by the user in case of a possible malfunction.
 * Subject to redirection, normally on stderr.
 */
int printfWarning(const char *format, ...) {
	if (verbosity>VERBOSITY_SILENT_ERROR) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

/** Prints out error messages normally expected by the user in case of a malfunction.
 * Subject to redirection, normally on stderr.
 */
int printfError(const char *format, ...) {
	if (verbosity>VERBOSITY_SILENT_ERROR) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

