#include <fifo.h>
#include <fifoParse.h>
#include <c-linux/fd.h>
#include <c-linux/hexFile.h>
#include <stdlib.h>
#include <macros.h>

static char fifoInBuffer[1024]; static Fifo _fifoIn = { fifoInBuffer, sizeof fifoInBuffer, };
static char fifoOutBuffer[4*1024]; static Fifo _fifoOut = { fifoOutBuffer, sizeof fifoOutBuffer, };
static char fifoScanBuffer[1024]; static Fifo _fifoScan = { fifoScanBuffer, sizeof fifoScanBuffer, };
static char fifoUcInBuffer[1024]; static Fifo _fifoUcIn = { fifoUcInBuffer, sizeof fifoUcInBuffer, };
static char fifoUcOutBuffer[1024]; static Fifo _fifoUcOut = { fifoUcOutBuffer, sizeof fifoUcOutBuffer };
static int fdSerial = -1;

Fifo
	*fifoIn = &_fifoIn,		// Console -> prog
	*fifoOut = &_fifoOut,		// proc -> Console
	*fifoErr = &_fifoOut;		// proc -> Console
static Fifo
	*fifoScan = &_fifoScan,		// uC -> prog
	*fifoUcIn = &_fifoUcIn,		// uC -> prog
	*fifoUcOut = &_fifoUcOut;	// prog -> uC

#include <fifoPrint.h>

bool fifoPrintCharEscaped(Fifo *fifo, char c) {
	if (0x20<=c && c<=127
	|| c=='\n' || c=='\r') return fifoPrintChar(fifo,c);
	else return fifoPrintString(fifo,"\\x") && fifoPrintHex(fifo,(int)c & 0xFF,2,2);
}

void doIo(void) {
	// uC -> prog
	if (fifoCanWrite(fifoUcIn) && fdCanRead(fdSerial)) fifoWrite(fifoUcIn,fdRead(fdSerial));

	// prog -> uC
	while (fifoCanRead(fifoUcOut) && fdCanWrite(fdSerial)) fdWrite(fdSerial,fifoRead(fifoUcOut));

	// prog -> console
	while (fifoCanRead(fifoOut) && fdCanWrite(1)) fdWrite(1,fifoRead(fifoOut));


	// console -> prog
	if (fifoCanWrite(fifoIn) && fdCanRead(0)) {
		const int c = fdRead(0);
		if (c>=0) fifoWrite(fifoIn,(char)c);
		else exit(0);		// EOF program termination
	}
}

void flushOut(void) {
	while (fifoCanRead(fifoOut)) fdWrite(1,fifoRead(fifoOut));
}

bool sendCommand(const unsigned char *cmd, int n) {
	if (n<=fifoCanWrite(fifoUcOut)) {
		fifoWriteN(fifoUcOut,cmd,n);
		while (fifoCanRead(fifoUcOut)) doIo();
		return true;
	}
	else return false;
}

bool receiveCommand(unsigned char *cmd, int n) {
	while (fifoCanRead(fifoUcIn) < n) doIo();
	fifoReadN(fifoUcIn,cmd,n);
	return true;
}

#include <fifoPrint.h>

#define FORCE(x) if (!(x)) {\
	fprintf(stderr,"Program error, failed condition: %s\n",#x);\
	exit(1);\
	} else ;

#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <c-linux/serial.h>

const char* ramloader = "ramloader";

void ramloaderSync(void) {
	// flush input
	fifoSkipRead(fifoUcIn, fifoCanRead(fifoUcIn));

	unsigned char response = 0;
	const unsigned char test = 0xA0;

	printf("<sync>"); fflush(stdout);
	unsigned char cmdSync[] = { 5, test };
	FORCE(sendCommand(cmdSync, sizeof cmdSync));
	//<= fifoCanWrite(fifoUcOut));
	//fifoWriteN(fifoUcOut,cmdSync,sizeof cmdSync);

	FORCE(receiveCommand(&response, sizeof response));

/*	fifoPrintString(fifoOut,"Response=0x");
	fifoPrintHex(fifoOut,response,2,2);
	fifoPrintString(fifoOut,"='");
	fifoPrintChar(fifoOut,response);
	fifoPrintChar(fifoOut,'\'');
	fifoPrintLn(fifoOut); flushOut();
*/	FORCE((~response & 0xFF)==test);
//	fifoPrintString(fifoOut,"Sync OK.\n");
}

/** Calculates the checksum of a 256 byte block.
 */
Uint16 checksum(const Uint8 *data) {
	Uint16 sum = 0;
	for (int i=0; i<256; ++i) sum += (Uint16)data[i] & 0xFF;
	return sum;
}

bool downloadCode(const char *fnImage, unsigned address, bool verify) {
	Uint8	ram[0x20000];
	HexImage hexImage = { ram, sizeof ram };
	hexImageInit(&hexImage);

	const unsigned char cmdResetAddress = 0x1;
	const unsigned char cmdWrite = 0x2;
	const unsigned char cmdSetAddress = 0x7;	// + 4 bytes absolute address.
	const unsigned char cmdExecute = 0x4;
	fifoPrintString(fifoOut,"<Downloading image>\n");
	ramloaderSync();		// armloader lost sync at this point for a long time.
	if (hexFileLoad(fnImage,&hexImage)) {
		for (int s=0; s<=hexImageSegments(&hexImage); s++) {
			const Segment *segment = &hexImage.segments[s];
			fifoPrintSegment(fifoOut,segment);
			fifoPrintString(fifoOut," :");

			if (addressIsValid(segment->address)) {	// send setAddress
				const unsigned char setAddress[] = {
					cmdSetAddress,
					segment->address & 0xFF,
					segment->address>>8 & 0xFF,
					segment->address>>16 & 0xFF,
					segment->address>>24 & 0xFF
				};
				FORCE(sendCommand(setAddress, sizeof setAddress));
				unsigned char response;
				FORCE(receiveCommand(&response,sizeof response));
				FORCE(response=='A');
			}
			else if (s==0) {
				FORCE(sendCommand(&cmdResetAddress, sizeof cmdResetAddress));
				unsigned char response;
				FORCE(receiveCommand(&response,sizeof response));
				FORCE(response==0xA5);
			}
			else {
				fifoPrintString(fifoOut,"Missing segment address.\n");
				return false;
			}

			for (int b=0; b<segment->size; b+=256) {
				Uint8 block[256];
				memset(block,0xFF,sizeof block);
				memcpy(block,segment->data+b,segment->size-b >= 256 ? 256 : segment->size-b);
				FORCE(sendCommand(&cmdWrite, sizeof cmdWrite));
				FORCE(sendCommand(block, sizeof block));
				const Uint16 sum;
				FORCE(receiveCommand((unsigned char*)&sum,sizeof sum));
				fifoPrintChar(fifoOut,'.');
			/*	fifoPrintString(fifoOut,"sum=0x");
				fifoPrintHex(fifoOut,sum,4,4);
				fifoPrintString(fifoOut,", sum(block)=0x");
				fifoPrintHex(fifoOut,checksum(block),4,4);
				fifoPrintLn(fifoOut);
			*/
				flushOut();
				FORCE(sum==checksum(block));
			}
			fifoPrintStringLn(fifoOut,"OK.");
		}
		fifoPrintStringLn(fifoOut,"<Executing code>.");
		FORCE(write(fdSerial,&cmdExecute,1));
		return true;
	}
	else {
		fifoPrintString(fifoOut,"Cannot load image.\n");
		return false;
	}
}

#if 0
void downloadSegments(const Segment *segments) {
	unsigned char buf[256];
	const unsigned char cmdResetAddress = 0x1;
	const unsigned char cmdWrite = 0x2;
	const unsigned char cmdExecute = 0x4;
	const unsigned char cmdSetAddress = 0x7;	// + 4 bytes absolute address.

	fifoPrintString(fifoOut,"<Downloading image >\n");
	ramloaderSync();		// armloader lost sync at this point for a long time.
	
	FORCE(sendCommand(&cmdResetAddress,sizeof cmdResetAddress));
	unsigned char response;
	FORCE(receiveCommand(&response,sizeof response));
	FORCE(response==0xA5);
	
	unsigned n, nTotal = 0, nPaged = 0;
	do {
		memset(buf,0,sizeof buf);
		n = read(fdImage,buf,sizeof buf);
		if (n>0) {
			FORCE(sendCommand(&cmdWrite, sizeof cmdWrite));
			FORCE(sendCommand(buf,sizeof buf));
			unsigned short sum,mysum = 0;
			for (int i=0; i<sizeof buf; i++) mysum += buf[i];
			FORCE(receiveCommand((unsigned char*)&sum,sizeof sum));
			fifoPrintChar(fifoOut,'.');
			//if (sum!=mysum) fprintf(stderr,"sum=0x%x, mysum=0x%x\n",sum,mysum);
			FORCE(sum==mysum);
		}
		nTotal += n;
		nPaged += sizeof buf;
	} while (n==sizeof buf);

	fifoPrintString(fifoOut,"\n<Done. Size=");
	fifoPrintUDec(fifoOut,nTotal,1,6);
	fifoPrintString(fifoOut,"B >\n");

#if 0
	if (verify) {
		cout << ramloader << ":verifying downloaded code:" << endl << ramloader << ':' << flush;
		lseek(fdImage,0,SEEK_SET);	// rewind file

		// rewind ramloader
		FORCE(sizeof cmdResetAddress==write(fdSerial,&cmdResetAddress,sizeof cmdResetAddress));
		unsigned char response;
		FORCE(1==read(fdSerial,&response,1));
		FORCE(response==0xA5);
	
		
		for (int o=0; o<nPaged; o+=sizeof buf) {
			memset(buf,0,sizeof buf);
			read(fdImage,buf,sizeof buf);

			FORCE(1==write(fdSerial,&cmdRead,1));
			for (int b=0; b<sizeof bufVerify; ++b) FORCE(1==read(fdSerial,&bufVerify[b],1));
			for (int b=0; b<sizeof bufVerify; ++b) {
				if (buf[b]!=bufVerify[b]) {
					cout << "incorrect byte at offset 0x" << hex << (o+b) << dec << endl;
					throw "transmission error";
				}
			}
			cout << '#' << flush;
		}
		cout << endl << "OK." << endl;
	}
#endif
	close(fdImage);
	FORCE(write(fdSerial,&cmdExecute,1));
}
#endif

int main(int argc, char* argv[]) {
	const char *envTty = "ARM_TTY";
	struct {
		const char *port;
		unsigned baud;
		unsigned address;
		unsigned timeoutMs;
		bool verify;
	}
	options = {
		getenv(envTty) ? getenv(envTty) : "/dev/ttyUSB0",
		115200,
		0x40000000,
		false,		// do not verify by default
	};

	for (char optChar; -1!=(optChar = getopt(argc,argv,"b:d:t:hv?")); ) switch(optChar) {

		case 'd':	options.port = optarg; break;
		case 'b':	options.baud = strtol(optarg,0,0); break;
		case 'v':	options.verify = true; break;
		case 't':	options.timeoutMs = strtol(optarg,0,0); break;
		case 'h':
		case '?':
		default :
			printf("%s 2.0 (formerly known as armloader), (C) Marc Prager, 2005-2012\n",ramloader);
			printf("NXP LPC1xxx/2xxx RAM loader program\n");
			printf("usage: %s [options] <image-file>\n",ramloader);
			printf("<image-file> may be binary or intel-hex in case of extension .hex\n");
			printf("options:\n");
			printf("  -b <baud rate>    : serial communication speed [%d]\n",options.baud);
			printf("  -d <device>       : default serial communication device [%s",options.port);
			printf("] optionally set by variable %s\n",envTty);
			printf("  -t <timeout/ms>   : serial port receive timeout\n");
			printf("  -v                : verify downloaded image\n");
			printf("  -h or -?          : help\n\n");
			return 1;
	}

	if (optind>=argc) {
		fprintf(stderr,"%s: missing input file. Try option: -?\n",ramloader);
		return 1;
	}
	
	const char *fileName = argv[optind];

	//printf("%s:tty=%s\n",ramloader,options.port);
	fdSerial = serialOpenBlockingTimeout(options.port,options.baud,options.timeoutMs/100);
	FORCE(fdSerial>=0);

	const char *pattern = "<RAMLOADER: hardware reset>\n";
	
	// disable BSL=RTS on Olimex LPC21-boards and AEGMIS-GIM1/2
	serialSetRts(fdSerial,false);
	serialSetDtr(fdSerial,1);
	usleep(200*1000);
	serialSetDtr(fdSerial,0);
	usleep(200*1000);

	(void)fifoScan;
	while (true) {
		doIo();

		if (fifoCanRead(fifoUcIn)
		&& 4<=fifoCanWrite(fifoOut)
		&& fifoCanWrite(fifoScan)) {
			const char c = fifoRead(fifoUcIn);
			fifoWrite(fifoScan,c);
			fifoPrintCharEscaped(fifoOut,c);
			if (fifoSearch(fifoScan,pattern)) {
				downloadCode(fileName,options.address,options.verify);
			}
			// restart scan.
			// reset input
		}

		if (fifoCanRead(fifoIn) &&fifoCanWrite(fifoUcOut)) fifoWrite(fifoUcOut, fifoRead(fifoIn));
	}

#if 0
	bool run = true;
	while (run) {
		FD_ZERO(&readFds);
		FD_ZERO(&writeFds);
		FD_SET(fdSerial,&readFds);
		FD_SET(0,&readFds);
		select(fdSerial+1,&readFds,&writeFds,NULL,NULL);
	
		if (FD_ISSET(fdSerial,&readFds)) {
			int n = read(fdSerial,buf,sizeof buf);
			FORCE(n>=0 /* read */);
			writeChars(buf,n);
			FORCE(fifoCanWrite(fifoIn)>=n);
			FORCE(fifoCanWrite(fifoScan)>=n);
			fifoWriteN(fifoIn,buf,n);
			fifoWriteN(fifoScan,buf,n);
			FORCE(fifoPrintNEscaped(fifoOut,buf,n));

			if (fifoSearch(fifoScan,pattern)) {
				fifoCopyReadPosition(fifoIn,fifoScan);	// skip detection pattern.
				download_code(fdSerial,fileName,options.address,options.verify);
			}
		}

		if (FD_ISSET(0,&readFds)) {	// keyboard input
			int n = read(0,buf,sizeof buf);
			FORCE(n==write(fdSerial,buf,n));
			run = n!=0;
		}

		
	}
#endif

	close(fdSerial);
	return 0;
}
