#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <stdlib.h>
#include <stdbool.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <c-linux/serial.h>
#include <fifoPopt.h>

const char* histFn = "/tmp/uconsole.hist";

void showHelp(void) {
	fprintf(stderr,
	"Usage: uconsole [options]\n"
	"  -?,-h                    : show this help\n"
	"  -b <baud>                : baud rate [115200]\n"
	"  -d <device>              : serial device [/dev/ttyUSB0]\n"
	"  -s <wave>                : control RTS and DTR (GIMx BSL and RESET)\n"
	"  -t <timeout/ms>          : receiver timeout [100]\n"
	"  -c                       : show control chars as escape sequences\n"
	"  -C                       : show all control chars, even \\r and \\n\n"
	"  -l {CR|LF|CRLF}          : transmission line feed [LF]\n"
	"  -p <prompt>              : show a prompt\n"
	"  -o <outputDevice>        : send received data to <outputDevice> [stdout]\n"
	"\n"
	"<wave> is a sequence of the chars rRdDpP with the following meanings:\n"
	"  r : RTS=0, R : RTS=1\n"
	"  d : DTR=0, D : DTR=1\n"
	"  p : pause 100ms\n"
	);
}

static char fifoCmdLineBuffer[1024];
static Fifo fifoCmdLine = { fifoCmdLineBuffer, sizeof fifoCmdLineBuffer };

static bool helpShow = false;
static bool controlCharsShow = false;
static bool controlCharsShowAll = false;
static Int32 baud = 115200;
static Int32 timeoutMs = 100;
static int symbolLineFeed = 2;
static Fifo fifoDevice = {};
static Fifo fifoWave = {};
static Fifo fifoPrompt = {};
static Fifo fifoOutputDevice = {};

const FifoPoptBool optionBools[] = {
	{ '?', &helpShow },
	{ 'h', &helpShow },
	{ 'c', &controlCharsShow },
	{ 'C', &controlCharsShowAll },
	{}
};

const FifoPoptInt32 optionInt32s[] = {
	{ 'b', &baud, },
	{ 't', &timeoutMs, },
	{}
};

const char* lineFeedSymbols[] = {
	"CR","LF","CRLF",
	0
};

const FifoPoptSymbol optionSymbols[] = {
	{ 'l', &symbolLineFeed, lineFeedSymbols },
	{}
};

const FifoPoptString optionStrings[] = {
	{ 'd', &fifoDevice },
	{ 's', &fifoWave },
	{ 'p', &fifoPrompt },
	{ 'o', &fifoOutputDevice },
	{}
};

bool parseCmdLine(int argc, const char* const *argv) {
	if (fifoPoptAccumulateCommandLine(&fifoCmdLine,argc,argv)) {
		while (fifoCanRead(&fifoCmdLine)) {
			if (fifoPoptBool(&fifoCmdLine,optionBools)
			|| fifoPoptInt32(&fifoCmdLine,optionInt32s)
			|| fifoPoptString(&fifoCmdLine,optionStrings)
			|| fifoPoptSymbol(&fifoCmdLine,optionSymbols)
			) ;	// all fine
			else {
				fprintf(stderr,"Invalid argument: %s", fifoReadPositionToString(&fifoCmdLine));
				return false;
			}
		}
		return true;
	}
	else return false;
}

typedef struct {
	int		readFd;
	int		writeFd;
	volatile bool	run;
} TControl;

void* dumpInput(TControl *tc) {
	char c;
	while (tc->run) {
		const int n=read(tc->readFd,&c,1);
		switch(n) {
			case 0: break;
			case 1: write(tc->writeFd,&c,1); break;
			default: tc->run = false;
		}
	}
	return 0;
}


int main(int argc, const char* const*argv) {
	if (!parseCmdLine(argc,argv)) return 1;
	if (helpShow) {
		showHelp();
		return 0;
	}
	const char *envDefault = getenv("ARM_TTY");
	const char *device = fifoIsValid(&fifoDevice) ? fifoReadPositionToString(&fifoDevice)
		: ( envDefault!=0 ? envDefault : "/dev/ttyUSB0");
		
	const int fd = serialOpenBlockingTimeout(device,baud,timeoutMs/100);
	if (fd<0) return 1;

	int fdOut = 1;
	if (fifoIsValid(&fifoOutputDevice)) {
		const char *terminal = fifoReadPositionToString(&fifoOutputDevice);
		fdOut = open(terminal, O_WRONLY | O_CREAT | O_TRUNC,
			S_IRUSR|S_IWUSR | S_IRGRP|S_IWGRP | S_IROTH|S_IWOTH );
		if (fdOut<0) {
			fprintf(stderr,"Error opening \"%s\"\n",terminal);
			return 1;
		}
	}

	TControl tc = { fd,fdOut,true };
	pthread_t t;
	if (0!=pthread_create(&t,0,(void*((*)(void*)))&dumpInput,&tc)) return 1;

	if (fifoIsValid(&fifoWave)) {
		while (fifoCanRead(&fifoWave)) {
			const char c = fifoRead(&fifoWave);
			switch(c) {
			case 'r': serialSetRts(fd,false); break;
			case 'R': serialSetRts(fd,true); break;
			case 'd': serialSetDtr(fd,false); break;
			case 'D': serialSetDtr(fd,true); break;
			case 'p': usleep(100*1000); break;
			default :	fprintf(stderr,"Invalid char '%c' in wave definition.\n",c);
					return 1;
			}
		}
	}

	using_history();
	history_truncate_file(histFn,10);
	read_history(histFn);

	const char *prompt = fifoIsValid(&fifoPrompt) ? fifoReadPositionToString(&fifoPrompt) : 0;
	const char* lfs[] = { "\r","\n","\r\n" };
	char*	line;
	do {
		line = readline(prompt);
		if (line) {
			write(fd,line,strlen(line));
			write(fd,lfs[symbolLineFeed],strlen(lfs[symbolLineFeed]));
			add_history(line);
			free(line);
		}
	} while (line);
	fprintf(stderr,"Connection closed.\n");
	tc.run = false;
	write_history(histFn);
	return 0;
}

