#ifndef usb_h
#define usb_h

/** @file
 * @brief Common USB constants and structures.
 */

#include <stdbool.h>
#include <integers.h>
#include <fifoPrint.h>

/** Most device classes can only be used on the interface level, not the device level.
 */
enum UsbDeviceClass {
	USB_CLASS_0 =0x00,		///< indicates that the interfaces define the class rather than the device.
	USB_CLASS_AUDIO =0x01,			///< (interface)
	USB_CLASS_CDC =0x02,			///< (device and interface) Communication device class.
	USB_CLASS_HID =0x03,			///< (interface) human interface device
	USB_CLASS_PHYSICAL =0x05,		///< (interface)
	USB_CLASS_IMAGE =0x06,			///< (interface)
	USB_CLASS_PRINTER =0x07,		///< (interface)
	USB_CLASS_MSD =0x08,			///< (interface) mass storage devices
	USB_CLASS_HUB =0x09,			///< (device) USB hub
	USB_CLASS_CDC_DATA =0x0A,		///< (interface)
	USB_CLASS_SMART_CARD =0x0B,		///< (interface)
	USB_CLASS_CONTENT_SECURITY=0x0D,	///< (interface)
	USB_CLASS_VIDEO =0x0E,			///< (interface)
	USB_CLASS_PERSONAL_HEALTHCARE =0x0F,	///< (interface)
	USB_CLASS_DIAGNOSTIC_DEVICE =0xDC,	///< (device and interface)
	USB_CLASS_WIRELESS_CONTROLLER =0xE0,	///< (interface)
	USB_CLASS_MISCELLANEOUS =0xEF,		///< (device and interface)
	USB_CLASS_APPLICATION_SPECIFIC =0xFE,	///< (interface)
	USB_CLASS_VENDOR_SPECIFIC =0xFF,	///< (device and interface)
};

bool fifoPrintUsbDeviceClass(Fifo *fifo, int deviceClass);

typedef struct __attribute__((packed)) {
	Uint8	type;		///< direction [7] (1=in), type [6:5] and recipient [3:0]
	Uint8	request;	///< UsbRequestType
	Uint16	value;	
	Uint16	index;
	Uint16	length;
} UsbDeviceRequest;

enum UsbRequestType {
	USB_TYPE_STANDARD,
	USB_TYPE_CLASS,
	USB_TYPE_VENDOR,
};

enum UsbRequestCode {
	USB_GET_STATUS=0,
	USB_CLEAR_FEATURE,
	USB_RESERVED_1,
	USB_SET_FEATURE,
	USB_RESERVED_2,
	USB_SET_ADDRESS,
	USB_GET_DESCRIPTOR,
	USB_SET_DESCRIPTOR,
	USB_GET_CONFIGURATION,
	USB_SET_CONFIGURATION,
	USB_GET_INTERFACE,
	USB_SET_INTERFACE,
	USB_SYNC_FRAME,
};

enum UsbDescriptorType {
	USB_DESCRIPTOR_DEVICE=1,
	USB_DESCRIPTOR_CONFIGURATION,
	USB_DESCRIPTOR_STRING,
	USB_DESCRIPTOR_INTERFACE,
	USB_DESCRIPTOR_ENDPOINT,
	USB_DESCRIPTOR_DEVICE_QUALIFIER,
	USB_DESCRIPTOR_OTHER_SPEED_CONFIGURATION,
	USB_DESCRIPTOR_INTERFACE_POWER,
};

bool fifoPrintUsbDescriptorType(Fifo *fifo, int descriptorType);

/** The device descriptor provides general information about the device.
 * Exactly one device descriptor exists for a device.
 */
typedef struct __attribute__((packed)) {
	Uint8	length;		///< size of this data structure
	Uint8	type;		///< descriptor type
	Uint16	usb;		///< USB version as BCD. 2.0 -> 0x200
	Uint8	deviceClass;
	Uint8	deviceSubclass;
	Uint8	deviceProtocol;
	Uint8	maxPacketSize0;	///< endpoint 0 packet size: 8,16,32 or 64.
	Uint16	idVendor;
	Uint16	idProduct;
	Uint16	release;	///< device release number as BCD
	Uint16	indexManufacturer;
	Uint16	indexProduct;
	Uint16	indexSerialNumber;
	Uint8	configurations;	///< number of possible configurations
} UsbDeviceDescriptor;

bool fifoPrintUsbDeviceRequest(Fifo *fifo, UsbDeviceRequest const *packet);

#endif
