/*
  mxli.c - main program of mxli, a NXP LPC ARM controller ISP programmer for Linux.
  Copyright 2011-2013 Marc Prager
 
  mxli is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  mxli is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with mxli.
  If not see <http://www.gnu.org/licenses/>
 */

// C standard
#include <fcntl.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>

// *nix specific
#include <termios.h>
#include <unistd.h>
#include <time.h>		// high resolution timers

// c-any library headers
#include <fifoParse.h>
#include <fifoParseStructure.h>
#include <fifoPopt.h>
#include <fifoPrint.h>
#include <lpcMemories.h>
#include <uu.h>
#include <macros.h>
#include <int32Math.h>

// my own linux-specific functions
#include <c-linux/serial.h>
#include <c-linux/hexFile.h>

/* History:
 * Version 1.0 (09/2011)
 * ======================
 * - full functionality
 *
 * Version 1.1 (04/2012)
 * ======================
 * - added Intel hex-file support, although in a not very clean programming style.
 * - put under GPL (03/2013)
 *
 * Version 2.0 (07/2013)
 * ======================
 * - added support for LPC800 (binary ISP protocol)
 * - clean up of input functions
 * - changed options, many new options.
 * - added execute by RESET as universal start code method.
 * - added verbosity level selection.
 * - improved scripting versality by adding silencing options -z and -Z.
 * - added detailed debugging output
 * - added flow control functionality
 * LPC800 NO_ISP mode NOT supported.
 *
 * Version 2.1 (08/2013)
 * ======================
 * - fixed Read bug (image offset +1, a leading \n).
 * - fixed command-line parsing flaws (accumulated options unrecognized sometimes)
 * - implemented option -o <outputFile>
 * - implemented 'read CRC checksum' -s
 * - implemented blank check, -1
 * - support for LPC800 NO_ISP as CRP4.
 * - experimental support for LPC1800 (multi flash bank) ISP protocol.
 *
 * TODO:
 *   return codes should be more expressive that just yes/no - especially with the different wave forms at exit.
 *
 * Version 2.2
 * ============
 * - environment variable MXLI_PARAMETERS is put in front of command-line parameters
 * - option -J<n> defines the number of device IDs to read (command 'J')
 * - command-line switch for checksum vectors: -5 -> slot 5 (valid exception on Cortex!), -7 -> slot 7 (FIQ on ARM7!)
 * - based on c-any-gpl-trunk, NOT c-any-gpl-1.0, which was left behind by development.
 * - LPC11Exx included
 * - LPC13xx still missing in lib.
 */

#define ISP_VERSION_COPYRIGHT "2.2 ($Rev: 90 $) (C) Marc Prager 2011-2014."
#define ISP "mxli"

static inline bool lpcReset(int fd, bool active) {
	return serialSetDtr(fd,active);
}

static inline bool lpcBsl(int fd, bool active) {
	return serialSetRts(fd,active);
}

static char outBuffer[16384];
static char inBuffer[16384];
Fifo fifoOut = { outBuffer, sizeof outBuffer };
Fifo fifoIn = { inBuffer, sizeof inBuffer };

char lineChar = 0;	// last line separator inserted into fifoIn.

static bool useEcho = false;
static int subBlockSize = 64;		// for binary protocol only
static int commandJAnswerNumbers = 1;

const char* escape(char c) {
	//static char buffer[5];
	static char buffer[20];
	if (c=='\r') return "\\r";
	else if (c=='\n') return "\\n";
	else if (0x20<=c && c<=127) {
		buffer[0] = c;
		buffer[1] = 0;
		return buffer;
	}
	else {
		sprintf(buffer,"\\x%02X",c & 0xFF);
		return buffer;
	}
}

#include <stdarg.h>
/** This function writes to stdout for result processing (program output).
 * It is not subject to verbosity level or message redirection. This is
 * stdout.
 */
int printfOut(const char *format, ...) {
	va_list args;
	va_start(args,format);
	const int n = vprintf(format,args);
	fflush(stdout);
	va_end(args);
	return n;
}

enum Verbosity {
	VERBOSITY_SILENT_ERROR,
	VERBOSITY_SILENT,
	VERBOSITY_NORMAL,
	VERBOSITY_DEBUG,
};

static Int32 verbosity = VERBOSITY_NORMAL;

/** Prints out debug messages not normally expected by the user.
 * Subject to redirection, normally on cerr.
 */
int printfDebug(const char *format, ...);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int printfNormal(const char *format, ...);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int vprintfNormal(const char *format, va_list args);

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on cerr.
 */
int printfWarning(const char *format, ...);

/** Prints out error messages normally expected by the user in case of a malfunction.
 * Subject to redirection, normally on cerr.
 */
int printfError(const char *format, ...);

////////////////////////////////////////////////////////////////////////////////////////////////////

enum {
	FIFO_OUT_MAX = 80,
};
// Useful error output diagnostics
// debugging output, 8-bit human readable.
void printfDebugFifo(Fifo fifo) {
	char c;
	for (int l=0; l<FIFO_OUT_MAX && fifoCanRead(&fifo); )
		l += printfDebug("%s",escape(fifoRead(&fifo)));
	if (fifoCanRead(&fifo)) printfDebug("\\[..%d more]",(int)fifoCanRead(&fifo));
}

void printfErrorFifo(Fifo fifo) {
	char c;
	for (int l=0; l<FIFO_OUT_MAX && fifoCanRead(&fifo); )
		l += printfError("%s",escape(fifoRead(&fifo)));
	if (fifoCanRead(&fifo)) printfError("\\[..%d more]",(int)fifoCanRead(&fifo));
}

// data output
// data output
void printFifo(FILE *dst, Fifo *fifo) {
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		fprintf(dst,"%c",c);
	}
}

// data output
bool ioWriteFifo(int fd, Fifo *fifo) {
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		if (1!=write(fd,&c,1)) {
			printfError("ERROR: IO failed on descriptor %d.\n",fd);
			return false;
		}
	}
	return true;
}

const char* returnMessage(int returnCode) {
	switch(returnCode) {
		case 0:		return "CMD_SUCCESS";
		case 1:		return "INVALID_COMMAND";
		case 2:		return "SRC_ADDR_ERROR";
		case 3:		return "DST_ADDR_ERROR";
		case 4:		return "SRC_ADDR_NOT_MAPPED";
		case 5:		return "DST_ADDR_NOT_MAPPED";
		case 6:		return "COUNT_ERROR";
		case 7:		return "INVALID_SECTOR";
		case 8:		return "SECTOR_NOT_BLANK";
		case 9:		return "SECTOR_NOT_PREPARED";
		case 10:	return "COMPARE_ERROR";
		case 11:	return "BUSY";
		case 12:	return "PARAM_ERROR";
		case 13:	return "ADDR_ERROR";
		case 14:	return "ADDR_NOT_MAPPED";
		case 15:	return "CMD_LOCKED";
		case 16:	return "INVALID_CODE";
		case 17:	return "INVALID_BAUD_RATE";
		case 18:	return "INVALID_STOP_BIT";
		case 19:	return "CODE_READ_PROTECTION";
		default:	return "UNDEFINED ERROR";
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Flow control hook


////////////////////////////////////////////////////////////////////////////////////////////////////

/** Provides a point for interrupting the program an let the user take control
 * @return true to continue, false to stop program
 */
bool flowControlHook(char command, const char* format, ...);

/** This function reads in the answer from the LPC into fifoIn.
 * It quickly reads in all characters up to an CR or LF. CR and LF are both represented by LF in the Fifo.
 * Two LF in direct succession are avoided. If no CR or LF is found, this function runs into a timeout that should
 * terminate this simple application.
 * @return true, if a line was read, false if timeout happened or buffer was too small.
 */
bool fifoLoadLine(int fd, Fifo *fifoIn) {
	char c;
	while (1==read(fd,&c,1)) {
		switch(c) {
			case 0x13:	// flow control: stop
			case 0x11:	// flow control: start
				printfError("ERROR: Flow control not implemented :-(\n");
				return false;
			case '\n':
			case '\r':
				if (!lineChar && c!=lineChar) {	// don't recognize different line chars in succession
					printfDebug("LINE BREAK:\\%c\n",c=='\n' ? 'n' : 'r');
					lineChar = c;
					if (fifoPrintChar(fifoIn,'\n')) return true;
					else {
						printfError("ERROR: input buffer overflow.\n");
						return false;
					}
				}
				else printfDebug("DISCARD:\\%c\n",c=='\n' ? 'n' : 'r'); // discard line separators in succession
				lineChar = c;	// remember
				break;
			default:
				if (!fifoPrintChar(fifoIn,c)) return false;
				lineChar = 0;
		}
	}
	return false;	// timeout
}

/** Reads in another line, only if fifo does not already contain one.
 */
bool fifoLoadLineOnDemand(int fd, Fifo *fifo) {
	// check, if we need reloading...
	if (!fifoContainsChar(fifo,'\n')) {
		// maybe we should throw away the current contents??

		if (fifoLoadLine(fd,&fifoIn)) {
			printfDebug("lpc:");
			printfDebugFifo(fifoIn);
			return true;
		}
		else {
			printfError("ERROR: no answer from lpc.\n");
			return false;
		}
	}
	else return true;	// already a line in Fifo - nothing to do
}

bool fifoSkipEmptyLine(Fifo *fifo) {
	if (fifoCanRead(fifo) && fifoLookAhead(fifo)=='\n') {
		fifoRead(fifo);	// skip empty line
		return true;
	}
	else return false;
}

/** Reads in another non-empty line, only if fifo does not already contain one.
 */
bool fifoLoadNonemptyLineOnDemand(int fd, Fifo *fifo) {
	do {
		if (!fifoLoadLineOnDemand(fd,fifo)) return false;
	} while(fifoSkipEmptyLine(fifo));
	return true;	// already a line in Fifo - nothing to do
}


/** This function reads in the fixed-size answer from the LPC into fifo.
 * If not enough characters are received, this function runs into a timeout that should terminate this simple
 * application.
 * @param n the number of bytes to receive.
 * @return true, if n characters where read, false if timeout happened.
 */
bool fifoLoadN(int fd, Fifo *fifo, int n) {
	char c;
	for (int i=0; i<n; i++) {
		if (1==read(fd,&c,1)) {
			lineChar = false;	// it's considered binary data.
			// there cannot be character-based flow control
			if (!fifoPrintChar(fifo,c)) {
				printfError("ERROR: Input buffer overflow.");
				return false;
			}
		}
		else {
			printfError("ERROR: fifoLoadN(): serial timeout. requested=%d, received=%d\n",n,i);
			return false;
		}
	}
	return true;
}

/** As LPC ISP programs are all bull-shit, we scan for the expected string within all the garbage instead of expecting
 * an EXACT answer. If it's found, the whole line is removed from the input. If
 */
bool fifoFindStringInLine(Fifo *input, const char *word) {
	Fifo clone = *input;
	if (fifoMatchUntilPattern(&clone,word) && fifoMatchUntilChar(&clone,'\n')) {
		fifoCopyReadPosition(input,&clone);
		return true;
	}
	else return false;
}

/** Tries to find a string and if it doesn't an error message is output.
 */
bool fifoFindStringInLineOrError(Fifo *input, const char *word) {
	Fifo clone = *input;
	if (fifoMatchUntilPattern(&clone,word) && fifoMatchUntilChar(&clone,'\n')) {
		fifoCopyReadPosition(input,&clone);
		return true;
	}
	else {
		printfDebug("ERROR: pattern \"%s\" not found, Fifo was \"",word);
		printfDebugFifo(clone);
		printfDebug("\")\n");

		return false;
	}
}

/** Reads in a line (if neccessary) that is not an echo line.
 * This function priorizes the echo!!
 */
bool lpcGetNonEchoAnswerLine(int fd, Fifo *fifo, const char *echo) {
	// check, if we need reloading...
	if (!fifoLoadLineOnDemand(fd,fifo)) return false;

	Fifo clone = *fifo;
	// check if it's an echo
	if (fifoFindStringInLine(fifo,echo)) {
		// yes it is, but it's already removed now.
		printfDebug(" - skipping echo line. (pattern was \"%s\", Fifo was \"",echo);
		printfDebugFifo(clone);
		printfDebug("\")\n");

		// load another, if neccessary...
		return fifoLoadLineOnDemand(fd,fifo);
	}
	else return true;	// not an echo
}

/** Consumes a line along with anything (left) in it.
 */
bool fifoConsumeLine(Fifo *fifo) {
	return fifoMatchUntilChar(fifo,'\n');
}

/** Reads one Int32 from a line and consumes the whole line, discarding anything else.
 */
bool fifoFindInt(Fifo *fifo, Int32 *result) {
	return	fifoParseInt(fifo,result)
		&& fifoConsumeLine(fifo);
}

/** Reads one Uint32 from a line and consumes the whole line, discarding anything else.
 */
bool fifoFindUnsigned(Fifo *fifo, Uint32 *result) {
	return	fifoParseUnsigned(fifo,result)
		&& fifoConsumeLine(fifo);
}

/** Reads the result code from a (zero return parameters) command.
 * Optionally the echo of LPC is skipped. The echo should be a pattern of the sent command.
 */
bool lpcFindResult(int fd, Fifo *fifo, const char* echo, Uint32 *result) {
	Fifo cloneBefore = *fifo;
	if (lpcGetNonEchoAnswerLine(fd,fifo,echo)
	&& fifoFindUnsigned(fifo,result)) {
		printfDebug(" - return code %s\n",returnMessage(*result));

		if (*result==0) return true;
		else {
			printfError("ERROR: %s\n",returnMessage(*result));
			return false;
		}
	}
	else {
		printfError("ERROR: could not read return code.\n");
		printfError("Input fifo before line load was \"");
		printfErrorFifo(cloneBefore);
		printfError("\"\nInput fifo after line load was \"");
		printfErrorFifo(*fifo);
		printfError("\"\n");
		return false;
	}
}

/** Reads n integer values after the result code was removed.
 */
bool lpcFindUnsignedValues(int fd, Fifo *fifo, Uint32 *values, int n) {
	for (int i=0; i<n; i++) {
		if (fifoLoadLineOnDemand(fd,fifo)
		&& fifoFindUnsigned(fifo,&values[i])) { // fine
			printfDebug(" - return value[%d]\n",i);
		}
		else {
			printfError("ERROR: could not read return value[%d]\n",i);
			return false;
		}
	}
	return true;
}

bool lpcWriteFifo(int fd, Fifo *fifo) {
	printfDebug(ISP ":");
	printfDebugFifo(*fifo);
	printfDebug("\n");
	while (fifoCanRead(fifo)) {
		const char c = fifoRead(fifo);
		flowControlHook('1',"1 single byte '%s' send.",escape(c));
		if (1!=write(fd,&c,1)) {
			printfError("ERROR: Timeout.\n");
			return false;
		}
	}
	return true;
}

bool lpcWriteString(int fd, const char *data) {
	if (fifoPrintString(&fifoOut,data)) return lpcWriteFifo(fd,&fifoOut);
	else {
		printfError("ERROR: Output buffer overflow.\n");
		return false;
	}
}

/** Maximum detailed diagnostics.
 */
bool lpcSync(int fd, int crystalHz) {
	if (!lpcWriteString(fd,"?")
	|| !fifoLoadLineOnDemand(fd,&fifoIn)) return false;

	if (fifoFindStringInLineOrError(&fifoIn,"Synchronized"))
		printfDebug(" - Accepted by "ISP"\n");
	else return false;

	if (!lpcWriteString(fd,"Synchronized\r\n")
	|| !fifoLoadLineOnDemand(fd,&fifoIn)) return false;

	// check, if we get something like "bla Synchronized bla\n"
	if (fifoFindStringInLineOrError(&fifoIn,"Synchronized"))
		printfDebug(" - Accepted by "ISP", but needs one more line with OK\n");
	else return false;

	if (fifoLoadLineOnDemand(fd,&fifoIn)
	&& fifoFindStringInLineOrError(&fifoIn,"OK"))
		printfDebug(" - Accepted by "ISP"\n");
	else return false;

	if (crystalHz > 0) {
		const Uint32 param = (crystalHz+500)/1000;
		fifoPrintUDec(&fifoOut,param,1,10); fifoPrintString(&fifoOut,"\r\n");
		Fifo clone = fifoOut;
		if (!lpcWriteFifo(fd,&fifoOut)) return false;

		if (!fifoLoadLineOnDemand(fd,&fifoIn)) return false;
		Uint32 result;
		if (fifoFindUnsigned(&fifoIn,&result)) {
			printfDebug(" - crystal frequency reply.\n");
		}
		else return false;

		if (result!=param) {
			printfError("ERROR: echo frequency mismatch.\n");
			return false;	// consume line containing crystal value
		}

		if (!fifoLoadLineOnDemand(fd,&fifoIn)) return false;
		if (fifoFindStringInLineOrError(&fifoIn,"OK"))
			printfDebug(" - Accepted by "ISP"\n");
		else return false;
	}

	return true;
}

bool lpcSendAndCheck(int fd, Fifo *fifoIn, const char *echo) {
	Uint32 result;
	bool ioSuccess =
		fifoPrintString(&fifoOut,"\r\n")
		&& lpcWriteFifo(fd,&fifoOut)
		&& lpcFindResult(fd,fifoIn,echo,&result);

	if (ioSuccess) {
		if (result==0) return true;
		else {
			printfError("ERROR: %s\n",returnMessage(result));
			return false;
		}
	}
	else return false;	// error message already provided
}

#define CE(fifo) checkEmpty(__LINE__,fifo);
void checkEmpty(int line, Fifo *fifo) {
	if (fifoCanRead(fifo)) {
		printfError("ERROR: line %d: Fifo not empty: \"%s\"",line,fifoReadLinear(fifo));
		abort();
	}
}

/** A command with no integer argument and n result integers.
 */
bool lpcCommand2Unsigneds(int fd, Fifo *fifoIn, char commandChar, Uint32 *values, int n) {
CE(fifoIn);
	char echoPattern[2] = { commandChar, 0 };
	return	fifoPrintChar(&fifoOut,commandChar)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern)
		&& lpcFindUnsignedValues(fd,fifoIn,values,n);
}

/** A command with 1 integer argument and zero return values.
 */
bool lpcCommandUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a) {
CE(fifoIn);
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);
}

/** A command with 2 integer arguments and zero return values.
 */
bool lpcCommandUnsignedUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a, Uint32 b) {
CE(fifoIn);
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,b,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);

}

/** A command with 3 integer arguments and zero return values.
 */
bool lpcCommandUnsignedUnsignedUnsigned(int fd, Fifo *fifoIn, char commandChar, Uint32 a, Uint32 b, Uint32 c) {
CE(fifoIn);
	char echoPattern[2] = { commandChar, 0 };

	return	fifoPrintChar(&fifoOut,commandChar)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,a,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,b,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintUDec(&fifoOut,c,1,10)
		&& lpcSendAndCheck(fd,fifoIn,echoPattern);

}

bool lpcBaud(int fd, int baud, int stopBits) {
	if (!flowControlHook('B',"B set baud rate to %d, %d stop bits.",baud,stopBits)) return false;
	return lpcCommandUnsignedUnsigned(fd,&fifoIn,'B',baud,stopBits);
}

// At least LPC17 answers A 0\r0\r\n on A 0\r\n. A \n is missing.
// Therefore we use only \r in this command.
// We turn echo off quickly to get rid of this shit.
bool lpcEcho(int fd, Fifo *fifoIn, bool on) {
	return lpcCommandUnsigned(fd,fifoIn,'A',(int)on);	
}

bool lpcReadBootCodeVersion(int fd, Fifo *fifoIn, Uint32 *version) {
	Uint32 values[2];
	if (!flowControlHook('K',"K read boot code version.")) return false;
	if (lpcCommand2Unsigneds(fd,fifoIn,'K',values,2)) {
		printfDebug("Boot code %u.%u\n",values[1],values[0]);
		*version = values[1]<<8 | values[0];
		return true;
	}
	else return false;
}

bool lpcReadPartId(int fd, Fifo *fifoIn, Uint32 *partIds) {
	switch(commandJAnswerNumbers) {
		case 0:		// do not use this command
		return true;

		case 1:
		if (!flowControlHook('J',"J read part ID (1 word).")) return false;

		if (lpcCommand2Unsigneds(fd,fifoIn,'J',partIds,1)) {
			printfDebug("Part ID 0x%08X\n",partIds[0]);
			return true;
		}
		else return false;

		case 2:
		if (!flowControlHook('J',"J read part ID (2 words).")) return false;

		if (lpcCommand2Unsigneds(fd,fifoIn,'J',partIds,2)) {
			printfDebug("Part IDs 0x%08X 0x%08X\n",partIds[0],partIds[1]);
			return true;
		}
		else return false;

		default:
			printfError("Internal Error.");
			return false;
	}
}

//Reads out the device serial number. @param sn 4 32-bit integers.
bool lpcReadUid(int fd, Fifo *fifoIn, Uint32 *uid) {
	if (!flowControlHook('N',"N read device UID.")) return false;
	return	lpcCommand2Unsigneds(fd,fifoIn,'N',uid,4);
}

bool lpcCopyRamToFlash(int fd, Fifo *fifoIn, unsigned addressFlash, unsigned addressRam, int n) {
	if (!flowControlHook('C',"C copy RAM 0x%08X to FLASH 0x%08X, size=%d.",addressRam,addressFlash,n)) return false;
	return	lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'C',addressFlash,addressRam,n);
}

bool lpcGo(int fd, Fifo *fifoIn, Uint32 pc, bool thumbMode) {
	const char echo[] = { 'G', 0 };

	if (!flowControlHook('G',"G execute address 0x%08X, mode %c.",pc,thumbMode ? 'T' : 'A')) return false;
	return	fifoPrintString(&fifoOut,"G ")
		&& fifoPrintUDec(&fifoOut,pc,1,10)
		&& fifoPrintChar(&fifoOut,' ')
		&& fifoPrintChar(&fifoOut,thumbMode ? 'T' : 'A')
		&& lpcSendAndCheck(fd,fifoIn,echo);
}

bool lpcPrepareForWrite(int fd, Fifo *fifoIn, int sectorStart, int sectorEnd, int flashBank) {
	if (flashBank==-1) {
		if (!flowControlHook('P',"P prepare for write sectors %d to %d.",sectorStart,sectorEnd)) return false;
		return	lpcCommandUnsignedUnsigned(fd,fifoIn,'P',sectorStart,sectorEnd);
	}
	else {
		if (!flowControlHook('P',"P prepare for write sectors %d to %d, flash bank %d.",
			sectorStart,sectorEnd,flashBank)) return false;
		return	lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'P',sectorStart,sectorEnd,flashBank);
	}
}

bool lpcErase(int fd, Fifo *fifoIn, int sectorStart, int sectorEnd, int flashBank) {
	if (flashBank==-1) {
		if (!flowControlHook('E',"E erase sectors %d to %d.",sectorStart,sectorEnd)) return false;
		printfNormal("Erasing sectors %d to %d NOW\n",sectorStart,sectorEnd);
		return	lpcCommandUnsignedUnsigned(fd,fifoIn,'E',sectorStart,sectorEnd);
	}
	else {
		if (!flowControlHook('E',"E erase sectors %d to %d, flash bank %d.",sectorStart,sectorEnd,flashBank))
			return false;
		printfNormal("Erasing sectors %d to %d, flash bank %d NOW\n",sectorStart,sectorEnd,flashBank);
		return	lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'E',sectorStart,sectorEnd,flashBank);
	}
}

bool lpcUnlock(int fd, Fifo *fifoIn) {
	const char echo[] = { 'U', 0 };
	if (!flowControlHook('U',"U unlock.")) return false;
	return	fifoPrintString(&fifoOut,"U 23130")
		&& lpcSendAndCheck(fd,fifoIn,echo);
}

bool lpcCompare(int fd, Fifo *fifoIn, Uint32 addrA, Uint32 addrB, int n) {
	if (!flowControlHook('M',"M compare memory 0x%08X and 0x%08X, length=%d.",addrA,addrB,n)) return false;
	if (lpcCommandUnsignedUnsignedUnsigned(fd,fifoIn,'M',addrA,addrB,n)) return true;
	else {
		printfDebug("Compare failed.\n");
		return false;
	}
}

bool lpcReadCrc(int fd, Fifo *fifoIn, Uint32 addr, int n, Uint32 *crc) {
	if (!flowControlHook('S',"S read CRC checksum, addr 0x%08X, length=%d.",addr,n)) return false;
	if (lpcCommandUnsignedUnsigned(fd,fifoIn,'S',addr,n)
	&& fifoLoadLineOnDemand(fd,fifoIn) && fifoFindUnsigned(fifoIn,crc)) return true;
	else {
		printfError("ERROR: read CRC checksum failed (is it supported by this LPC ?).\n");
		return false;
	}
}

bool lpcSetFlashBank(int fd, Fifo *fifoIn, int flashBank) {
	if (!flowControlHook('S',"S set active flash bank %d.",flashBank)) return false;
	if (lpcCommandUnsigned(fd,fifoIn,'S',flashBank)) return true;
	else {
		printfError("ERROR: cannot set active flash bank.\n");
		return false;
	}
}

/** lpcBlankCheck will not end in CMD_SUCCESS in case of non-blank sector(s). That's why we can't use
 * lpcCommandUnsignedUnsigned.
 */
bool lpcBlankCheck(int fd, Fifo *fifoIn, int _sectorStart, int sectorEnd, int flashBank, bool *blank, bool doBlankCheck0) {
	const char echo[] = { 'I', 0 };

	int sectorStart = doBlankCheck0 ? _sectorStart : int32Max(1,_sectorStart);

	if (doBlankCheck0 || _sectorStart>0)
		if (!flowControlHook('I',"I blank check sectors %d to %d.",sectorStart,sectorEnd)) return false;
		else ;
	else
		if (!flowControlHook('I',"I blank check sectors %d to %d. Sector 0 assumed to be non-blank.",
			sectorStart,sectorEnd)) return false;
		else ;

	if (_sectorStart==0 && !doBlankCheck0 && sectorEnd<1) {
		// nothing to do except assuming sector 0 is non-blank
		*blank = false;
		printfDebug("WARNING: Sector 0 assumed to be non-blank.\n");
		return true;
	}

	if (fifoPrintString(&fifoOut,"I ")
	&& fifoPrintUDec(&fifoOut,sectorStart,1,10)
	&& fifoPrintChar(&fifoOut,' ')
	&& fifoPrintUDec(&fifoOut,sectorEnd,1,10)
	&& (flashBank==-1 || fifoPrintChar(&fifoOut,' ') && fifoPrintUDec(&fifoOut,flashBank,1,10))
	&& fifoPrintString(&fifoOut,"\r\n")
	&& lpcWriteFifo(fd,&fifoOut)) ;	// fine
	else {
		printfError("ERROR: output buffer overflow or IO problem.\n");
		return false;
	}
	Uint32 result;
	if (!lpcGetNonEchoAnswerLine(fd,fifoIn,echo)) return false;

	if (fifoFindUnsigned(fifoIn,&result)) {
		printfDebug(" - return code %s\n",returnMessage(result));
		Uint32 offset,value;
		switch(result) {
		case 0:	*blank = _sectorStart>0 || doBlankCheck0;	// full check succeeded! (LPC800)
			if (!*blank) {
				printfDebug("WARNING: Sector 0 assumed to be non-blank.\n");
			}
			return true;
		case 8:	// SECTOR_NOT_BLANK \n <offset> \n <value>
			*blank = false;
			if (lpcFindUnsignedValues(fd,fifoIn,&offset,1)
			&&  lpcFindUnsignedValues(fd,fifoIn,&value,1)) {
				printfDebug("Sector not blank at sector[%d]+0x%08X, value=0x%08X\n",
					sectorStart,offset,value);
				return true;
			}
			else {
				printfError("ERROR: could not read offset and value.\n");
				return false;
			}
		default:
			printfError("ERROR: %s\n",returnMessage(result));
			return false;
		}
	}
	else {
		printfError("ERROR: could not read return code.\n");
		return false;
	}
}

bool lpcBailOut(int fd) {
	lpcWriteString(fd,"\x1B");
	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Read/Write functions
//
bool lpcReadUuencode(int fd, Fifo *fifoIn, Fifo *data, Uint32 address, Uint32 n) {
	const Uint32 r0 = fifoCanRead(data);

	if (lpcCommandUnsignedUnsigned(fd,fifoIn,'R',address,n)) {
		int checksum=0;
		for (int lineNo=0; fifoCanRead(data)-r0 <n; lineNo++) {
			if (fifoLoadLineOnDemand(fd,fifoIn)
			&& fifoUuDecodeLine(data,fifoIn,&checksum)
			&& fifoParseLineEnd(fifoIn))	// fine
				printfDebug("[line %d]\n",lineNo);	// for prettier output
			else return lpcBailOut(fd);

			if ((lineNo % 20)==19 || fifoCanRead(data)-r0 ==n) {
				Uint32 checksumLpc = 0;
				if (fifoLoadLineOnDemand(fd,fifoIn)
				&& fifoFindUnsigned(fifoIn,&checksumLpc)) {

					if (checksum==checksumLpc) {
						printfNormal(".");
						lpcWriteString(fd,"OK\r\n");	// :o) unchecked
					}
					else {
						printfError("ERROR: checksum invalid. 0x%x bytes, lpc:0x%08X, "ISP":0x%08X\n",
							fifoCanRead(data)-r0,
							checksumLpc,checksum);
						return lpcBailOut(fd);	// we do not retry
					}
					checksum = 0;
				}
				else {
					printfError("ERROR: could not read checksum from lpc.\n");
					return false;
				}
			}
		}
		return true;
	}
	else return false;
}

bool lpcReadBinary(int fd, Fifo *fifoIn, Fifo *data, Uint32 address, Uint32 n) {
	if (lpcCommandUnsignedUnsigned(fd,fifoIn,'R',address,n)) {
		// :o) There's always a \n in the output stream
		// It seems, LPC800 always prefixes the binary data with a \n
		if (fifoLoadN(fd,data,n+1)) {
			(void)fifoRead(data);	// discard the \n
			printfNormal(".");
			return true;
		}
		else return false;
	}
	else return false;
}

bool lpcRead(int fd, Fifo *fifoIn, int ispProtocol, Fifo *data, unsigned address, int n) {
	switch(ispProtocol) {
		case ISP_PROTOCOL_UUENCODE: return lpcReadUuencode(fd,fifoIn,data,address,n);
		case ISP_PROTOCOL_BINARY: return lpcReadBinary(fd,fifoIn,data,address,n);
		default: return lpcBailOut(fd);
	}
}

bool lpcWriteCommand(int fd, Fifo *fifoIn, Uint32 address, Uint32 size) {
	return lpcCommandUnsignedUnsigned(fd,fifoIn,'W',address,size);
}

/** Checked on LPC2136/01 -E and not -E.
 */
bool lpcWriteUuencode(int fd, Fifo *fifoIn, unsigned address, Fifo *data) {
	if (!flowControlHook('W',"W (write uuencode).")) return false;
	if (lpcWriteCommand(fd,fifoIn,address,fifoCanRead(data))) {
		int checksum=0;
		for (int lineNo=0; fifoCanRead(data); lineNo++) {
			if (!flowControlHook('w',"w (write uuencoded line).")) return false;
			if (fifoUuEncodeLine(&fifoOut,data,&checksum)) {
				//printfError("line %d\n",lineNo);
				lpcWriteFifo(fd,&fifoOut);
				if (useEcho) {
					if (!fifoLoadLineOnDemand(fd,fifoIn) || !fifoConsumeLine(fifoIn)) {
						printfError("ERROR: echo line missing.\n");
						return false;
					}
					printfDebug("\n");	// just formatting debug output
				}
			}
			else return lpcBailOut(fd);

			if ((lineNo%20)==19 || !fifoCanRead(data)) {	// checksum
				Uint32 checksumReturn;
				if (fifoPrintUDec(&fifoOut,checksum,1,10)
				&& fifoPrintString(&fifoOut,"\r\n")
				&& lpcWriteFifo(fd,&fifoOut)
				&& (!useEcho || lpcFindUnsignedValues(fd,fifoIn,&checksumReturn,1)
					&& checksum==checksumReturn)	// this checks communication only!
				&& fifoLoadLineOnDemand(fd,fifoIn)
				&& fifoFindStringInLineOrError(fifoIn,"OK")) {
					printfNormal(".");	// show progress, checksum confirmed
				}
				else {
					printfError("ERROR: checksum invalid.\n");
					return lpcBailOut(fd);
				}

				checksum = 0;
			}
		}
		return true;
	}
	else return false;
}

bool lpcWriteBinary(int fd, Fifo *fifoIn, unsigned address, Fifo *data) {
	const int n = fifoCanRead(data);
	if (!flowControlHook('W',"W (write binary).")) return false;
	if (lpcWriteCommand(fd,fifoIn,address,fifoCanRead(data))) {

		// data per write command
		printfNormal("[%u=",n);

		// binary data is always echoed
		// Send data in small blocks and receive answer
		// This is a primitive kind of flow control.
		const Uint32 chunkSize = subBlockSize;
		for (int o=0; o<n; o+= chunkSize) {
			const Uint32 cs = int32Min(chunkSize,n-o);

			Fifo fifoChunk;
			if (!fifoParseN(data,&fifoChunk,cs)) {
				printfError("ERROR: chunk extraction error.\n");
				return false;
			}

			if (!fifoPutFifo(&fifoOut,&fifoChunk)) {
				printfError("ERROR: output buffer overflow.\n");
				return false;
			}
			if (!flowControlHook('w',"w (write block[%d]).",(int)fifoCanRead(&fifoOut))) return false;
			if (!lpcWriteFifo(fd,&fifoOut)) {
				printfError("ERROR: IO error.\n");
				return false;
			}
			printfNormal("+%d",cs);

			if (!flowControlHook('r',"W (write block, read answer block[%d]).",(int)cs)) return false;
			if (useEcho) {
				if (fifoLoadN(fd,fifoIn,cs) && fifoCanRead(fifoIn)>=cs) {
					// we could compare here...
					fifoSkipRead(fifoIn,cs);
				}
				else {
					printfError("ERROR: binary echo mismatch.\n");
					printfError("Received data was: \"");
					printfErrorFifo(*fifoIn);
					printfError("\"\n");
					return false;
				}
			}
		}
		printfNormal("]");
		return true;
	}
	else return false;
}

bool lpcWrite(int fd, Fifo *fifoIn, int ispProtocol, unsigned address, Fifo *data) {
	switch(ispProtocol) {
		case ISP_PROTOCOL_UUENCODE: return lpcWriteUuencode(fd,fifoIn,address,data);
		case ISP_PROTOCOL_BINARY: return lpcWriteBinary(fd,fifoIn,address,data);
		default: return lpcBailOut(fd);
	}
}

bool lpcWriteN(int fd, Fifo *fifoIn, int ispProtocol, unsigned address, Fifo *data, int n) {
	Fifo fifoFraction;
	return	fifoParseN(data,&fifoFraction,n)
		&& lpcWrite(fd,fifoIn,ispProtocol,address,&fifoFraction);
}

typedef struct {
	bool 	verify;			///< verify flash after copy RAM to FLASH :o) 4 possibilities, not 2 (bool)!
	bool	fast;			///< waste FLASH lifetime for speed
	int	ramLowerSpace;		///< ISP uses 0x200 bytes at bottom of RAM, 0x270 on LPC800.
	int	ramUpperSpace;		///< ISP/IAP uses 32 + 0x100 bytes at top of RAM
	bool	doBlankCheck0;		///< omit (and always assume non-blank) sector 0
	bool	doVerifyLower;		///< do omit (and always assume correct) lower n bytes
	bool	doCrcLower;		///< do omit lower n bytes
	int	remapBytes;		///< number of re-mapped bytes, 64 typically
	int	fBank;			///< destination FLASH bank. -1 if unused.
} LpcProgramConfig;

/** Extended programming function for hex Images.
 *  @param fast erase sectors in one call, if not all blank. Otherwise erase sector for sector.
 * code read protection
 */
bool lpcProgram(int fd, Fifo *fifoIn, LpcFamilyMember const *member, const HexImage *hexImage, const LpcProgramConfig *conf) {
	const int blockLimit = member->sizeRamKs[0]*1024 - conf->ramLowerSpace - conf->ramUpperSpace;
	const LpcFamilyMemoryMap *map = member->familyMemoryMap;
	const int addressRam = map->addressRams[0] + conf->ramLowerSpace;
	if (blockLimit>=1024) printfNormal("Transfer RAM: %dkiB+%dB @ 0x%08X\n",blockLimit/1024,blockLimit%1024,addressRam);
	else printfNormal("Transfer RAM: 0x%03XB @ 0x%08X\n",blockLimit,addressRam);

	if (hexImageSegments(hexImage)!=1) {
		printfError("Cannot handle more than one segment so far, yet.\n");
		return false;
	}
	const int imageSize = hexImage->segments[0].size;
	const int address = hexImage->segments[0].address;

	// copy image to a Fifo
	char imageBuffer[512*1024];
	Fifo fifoImage = { imageBuffer, sizeof imageBuffer };
	for (int b=0; b<imageSize; b++) fifoWrite(&fifoImage,hexImage->segments[0].data[b]);

	// optimized for big images => maximum block with 'Copy RAM to FLASH' unless image is really small.
	int blockSize = 0;
	for (int i=0; i<ELEMENTS(map->blockSizes) && map->blockSizes[i]>0; ++i) {
		if (map->blockSizes[i] <= blockLimit
		&& blockSize<imageSize) 	blockSize = map->blockSizes[i];
	}
	if (blockSize==0) {
		printfError("(Remaining) RAM (0x%0XB) to small for Copy-RAM-To-FLASH.\n",blockLimit);
		return false;
	}
	// pad fifo for this block size
	while (fifoCanRead(&fifoImage) % blockSize) if (!fifoPrintChar(&fifoImage,0xFF)) return false;

	const int blocks = fifoCanRead(&fifoImage) / blockSize;
	const int blocksAtOnce = blockLimit / blockSize;	// the will be 1 at least

	printfNormal("Transmission RAM: %dblocks (%dkiB), units of %dB\n",
		blocksAtOnce, blocksAtOnce*blockSize/1024, blockSize);

	printfNormal("Flash address range 0x%08x..0x%08x, ",address,address+blockSize*blocks-1);
	int sectorFrom, sectorTo;
	if (lpcAddressRangeToSectorRange(member,&sectorFrom,&sectorTo,address,address+blockSize*blocks-1)) {
		printfNormal("sectors: %d..%d\n",sectorFrom,sectorTo);
	}
	else {
		printfError("ERROR: image does not fit into FLASH\n");
		return false;
	}

	if (!lpcUnlock(fd,fifoIn)) {
		printfError("ERROR: failed to unlock device.\n");
		return false;
	}

	// Manual of LPC800 states, that sector 0 cannot be blank checked (will always fail)
	// However, in reality, this seems to be wrong
	bool blank;
	if (!lpcBlankCheck(fd,fifoIn,sectorFrom,sectorTo,conf->fBank,&blank,conf->doBlankCheck0)) {	// erasing required
		printfError("ERROR: failed to perform blank ckeck.\n");
		return false;
	}

	if (!blank) {	// erasing required
		if (conf->fast) {	// erase all sectors, even if not all of them are non-blank
			if (lpcPrepareForWrite(fd,fifoIn,sectorFrom,sectorTo,conf->fBank)
			&& lpcErase(fd,fifoIn,sectorFrom,sectorTo,conf->fBank)) printfNormal("Erased en bloc.\n");
			else {
				printfError("ERROR: failed to erase sectors %d to %d\n",sectorFrom,sectorTo);
				return false;
			}
		}
		else for (int s=sectorFrom; s<=sectorTo; ++s) {
			if (!lpcBlankCheck(fd,fifoIn,s,s,conf->fBank,&blank,conf->doBlankCheck0)) {	// erasing required
				printfError("ERROR: failed to perform blank ckeck.\n");
				return false;
			}
			
			if (blank
			|| lpcPrepareForWrite(fd,fifoIn,s,s,conf->fBank) && lpcErase(fd,fifoIn,s,s,conf->fBank)) ;	// fine
			else {
				printfError("ERROR: failed to prepare/erase sector %d\n",s);
				return false;
			}
		}
	}

	for (int block=0; block<blocks; ) {
		// transfer bytes
		for (int subBlock=0; subBlock<blocksAtOnce && block+subBlock<blocks; ++subBlock) {
			printfNormal("\nTransfer block %d to RAM (%s) ",block+subBlock,
				member->ispProtocol==ISP_PROTOCOL_UUENCODE ? "uuencode" : "binary" );
			if (lpcWriteN(fd,fifoIn,member->ispProtocol,addressRam+subBlock*blockSize,&fifoImage,blockSize)) {	// fine
				printfNormal("OK.");
			}
			else {
				printfNormal("FAILED!\n");	// this is part of progress output
				printfError("ERROR: could not write block to RAM.\n");
				return false;
			}
		}
		// do flash programming
		for (int subBlock=0; subBlock<blocksAtOnce && block+subBlock<blocks; ++subBlock) {
			const Uint32 addressFlash = address+(block+subBlock)*blockSize;
			const int sector = lpcAddressToSector(member,addressFlash);
			printfNormal("\nTransfer block %d to FLASH, sector %d ",block+subBlock,sector);
			if (lpcPrepareForWrite(fd,fifoIn,sector,sector,conf->fBank)
			&& lpcCopyRamToFlash(fd,fifoIn, addressFlash, addressRam+subBlock*blockSize, blockSize)); // fine
			else {
				printfError("ERROR: failed to Copy RAM to FLASH.\n");
				return false;
			}
			if (conf->verify) {
				if (addressFlash==0) {
					printfNormal(" verify not possible for sector 0.");
				}
				else {
					if (lpcCompare(fd,fifoIn,addressFlash,addressRam+subBlock*blockSize, blockSize)) 
					printfNormal(" verify OK.");
					else {
						printfNormal(" verify failed.\n");	// this is part of progress output
						printfError("ERROR: Verification failed at address 0x%08X.\n",addressFlash);
						return false;
					}
				}
			}
		}
		block += blocksAtOnce;
	}
	printfNormal("\n");

	return true;
}

static inline int min32(int a, int b) {
	return a<b ? a : b;
}

/** Re-writes a fifo, so that the LPCxxxx checksum (first 8 vectors) is 0. Use: vector 5, offset 0x14.
 */
bool fifoLpcChecksum(Fifo *fifo, int reservedVector) {
	Fifo clone = *fifo;
	Fifo writer;
	fifoInitRewrite(&writer,fifo);

	Uint32 checksum = 0;
	for (int i=0; i<8; ++i) {
		Uint32 vector;
		if (!fifoGetUint32(&clone,&vector)) return false;
		if (i!=reservedVector) checksum += vector;
	}
	fifoSkipWrite(&writer,4*reservedVector);	// target vector
	return fifoPutInt32(&writer,-checksum);
}

// only correct for segment 0
bool lpcChecksumFix(const HexImage *hexImage, int reservedVector) {
	Fifo fifo;
	fifoInitRead(&fifo, hexImage->segments[0].data, hexImage->segments[0].size);
	if (hexImage->segments[0].address==0) return fifoLpcChecksum(&fifo,reservedVector);
	else return true;
}

static unsigned address = 0;
static Int32 baud = 230400;
static Int32 crystalHz = 14748000;	// IRC + PLL of LPC17 and LPC23
static Int32 addressCrp = 0x2FC;	// code read protection word address
static const char * const defaultDevice = "/dev/ttyUSB0";
static Fifo fifoDevice = {};
static int eraseSymbol = -1;
static Int32 flashSizeOverride = -1;	// flash size override
static int manualSymbol = -1;
static Int32 levelCrpForce = -1;	// set this level of protection
static Int32 levelCrpAllow = 0;		// limit the level of protection
static Int32 memorySizeOverride = -1;	// RAM 0 size override
static bool quick = false;		// prefer speed to flash lifetime
static Int32 readN = -1;
static bool showInfo = false;
static Int32 timeAfterResetMs = 200;
static Int32 timeoutSerialMs = 1000;	// 500 is enough for LPC17, but not the LPC21s
static bool verify = false;
static Int32 writeN = -1;		// write a specified number of bytes
static bool writeAll = false;		// write all
static Int32 executeAddress = -1;
static int executeSymbol = -1;
//static Int32 ucNumber = -1;		// force specific device
static int ucSymbol = -1;		// force user supplied.
static Fifo fifoF = {};
static Fifo fifoM = {};
static Fifo fifoX = {};			// execution control
static Fifo fifoN = {};			// controller name, provided by user
static Fifo fifoU = {};			// manually selected device
static Fifo fifoO = {};			// output File
static Fifo fifo0 = {};			// use sector 0 operations
static Fifo fifoW = {};			// -W waveform complex argument
static char defaultW[] = ".i=rdpDPR.x=dpD.j=.e=dp";	// default waveform definitions
static const char *manuallySelected = 0;// user provides processor name
static Int32 ispProtocolOverride = -1;	// binary / uuencode override
static PairInt32 erase = { 0, -1 };
// since at least LPC800, bottom ISP RAM requirement has increased from 0x200 to 0x270
static Int32 ispReserved[2] = { 0x270, 0x120 };	// RAM used by ISP handler, bottom/top of RAM0
static Int32 userId = -1;
static Int32 listB[LPC_BLOCK_SIZES] = { };
static bool showAllDevices = false;	// command
static bool probeBaud = false;		// command
static bool showUid = false;		// command
static bool showBoot = false;		// command
static bool noUcOutput = false;		// output control: avoid default info of detected device.
static bool executeReset = false;	// command: start code by reset.
static Int32 remapBytes = 64;		// number of bytes affected by re-mapping of ISP.
static bool blankCheck = false;		// command
static int setFlashBank = -1;		// command. 
static int setFlashBankSymbol = -1;	// symbolic flash bank
static int fBank = -1;	// flash bank selection
static Int32 checksumRange[2] = { -1,-1 };	// n,startAddr
static Int32 checksumRangeSymbol = -1;		// or symbolic
static int checksumVector = 7;		// Cortex-M selection

static const FifoPoptBool optionBools[] = {
	{ '1',&blankCheck },
	{ 'i',&showInfo },
	{ 'k',&showBoot },
	{ 'n',&noUcOutput },
	{ 'p',&probeBaud },
	{ 'q',&quick },
	{ 'S',&showAllDevices },
	{ 'u',&showUid },
	{ 'v',&verify },
	{ 'w',&writeAll },
	{ 'x',&executeReset },
	{ 'E',&useEcho },
	{}
};

static const FifoPoptInt32Flag optionIntFlags[] = {
	{ 'z',&verbosity, VERBOSITY_SILENT },
	{ 'Z',&verbosity, VERBOSITY_SILENT_ERROR },
	{ 'g',&verbosity, VERBOSITY_DEBUG },
	{ '5',&checksumVector, 5 },
	{ '7',&checksumVector, 7 },
	{}
};

static const FifoPoptInt32 optionInts[] = {
	{ '2',&setFlashBank, &fifoParseInt },
	{ 'a',(Int32*)&address, &fifoParseIntEng },
	{ 'b',&baud, },
	{ 'c',&crystalHz, &fifoParseIntEng },
	{ 'C',&addressCrp, },
	{ 'f',&flashSizeOverride, &fifoParseIntEng },
	{ 'A',&remapBytes },
	{ 'I',&userId, },
//	{ 'N',&userNumber, },
	{ 'l',&levelCrpForce, &fifoParseInt },
	{ 'L',&levelCrpAllow, &fifoParseInt },
	{ 'm',&memorySizeOverride, &fifoParseIntEng },
	{ 'r',&readN, &fifoParseIntEng },
	{ 't',&timeAfterResetMs, },
	{ 'T',&timeoutSerialMs, },
//	{ 'U',&ucNumber, },
	{ 'j',&executeAddress, },
	{ 'J',&commandJAnswerNumbers, },
	{ 'y',&subBlockSize, },
	{ 'Y',&fBank, },
	{}
};

static const char* const eraseSymbols[] = {
	"all",
	0
};

static const char* const executeSymbols[] = {
	"LMA","LMA+1",
	0
};

static const char* const ucSymbols[] = {
	"user",
	0
};

static const char* const manualSymbols[] = {
	"contents",
	"numeric",
	"new-devices",
	"crp",
	"flow-control",
	"questionable",
	"philosophy",
	"troubleshooting",
	"waveforms",
	0
};

static const char* const ispProtocolSymbols[] = {
	"UUENCODE","BINARY",		// these positions must match the enum im lpcMemories.h !
	0
};

static const char* const checksumRangeSymbols[] = {
	"file","flash",
	0
};

static const FifoPoptSymbol optionSymbols[] = {
	{ 'e',&eraseSymbol, eraseSymbols },
	{ 'h',&manualSymbol, manualSymbols },
	{ 'j',&executeSymbol, executeSymbols },
	{ 'U',&ucSymbol, ucSymbols },
	{ 'P',&ispProtocolOverride, ispProtocolSymbols },
	{ 's',&checksumRangeSymbol, checksumRangeSymbols },
	{}
};

static const FifoPoptString optionStrings[] = {
	{ '0', &fifo0 },	// questionable operations: use sector 0 for ...
	{ 'd', &fifoDevice },
	{ 'o', &fifoO },
	{ 'F', &fifoF },
	{ 'M', &fifoM },
	{ 'X', &fifoX },	// execution flow control
	{ 'N', &fifoN },	// command line chip name definition
	{ 'U', &fifoU },	// manual processor selection
	{ 'W', &fifoW },	// waveform definition
	{}
};

static const FifoPoptInt32Range optionInt32Ranges[] = {
	{ 'e', &erase },
	{}
};

static const FifoPoptInt32Tuple optionInt32Tuples[] = {
	{ 'R', 2, ispReserved, ",", },
	{ 'B', 4, listB, ",", &fifoParseIntEng },
	{ 's', 2, checksumRange, "@", &fifoParseIntEng },
	{}
};

// complex parsing section: options -F and -M: lists of pairs

// end of option validator
bool eoo(Fifo *fifo) {
	return 0==fifoCanRead(fifo);
}

const FifoParsePairInt parseSizeXCount = {
	.parseIntA = &fifoParseIntEng,
	.separator = "x",
	.parseIntB = &fifoParseIntCStyle,
	// no validator!
};

const FifoParsePairInt parseSizeAtAddress = {
	.parseIntA = &fifoParseIntEng,
	.separator = "@",
	.parseIntB = &fifoParseIntCStyle,
	// no validator!
};

const FifoParseListPairInt parseF = {
	.parsePair = &parseSizeXCount,
	.separator = ",",
	.validator = &eoo
};

const FifoParseListPairInt parseM = {
	.parsePair = &parseSizeAtAddress,
	.separator = ",",
	.validator = &eoo
};

PairInt listF[3];
PairInt listM[3];

static void showHelp(void) {
	// small letters are mostly actions and basic settings.
	// capital letters are mostly options modifying default behaviour.
	printfOut(
	ISP " " ISP_VERSION_COPYRIGHT ". GPLv3 license.\n"
	"Marc's exquisite LPC ISP command-line programmer for NXP ARM devices.\n\n"
	"Usage: "ISP" <options> [<inputFile>]\n"
	"Options: (values in [] show the defaults, *=not yet implemented)\n"
	"  -? or --help              : display this help screen.\n"
	"  -0 <q-command-list>       : use sector 0 even for questionable operations specified\n"
	"  -1                        : blank check device (= filled with all ones).\n"
	"  -2 <bank>|Y               : select flash bank <bank> or from parameter -Y\n"
	"  -5                        : use vector #5 for checksum (ARM7)\n"
	"  -7                        : use vector #7 for checksum (Cortex-M)\n"
	"  -a <destinationAddress>   : image destination address (LMA) (sector boundary!!) [0x%X]\n"
	"  -b <baud rate>            : communication baud rate [%d]\n"
	"  -c <f crystal/Hz>         : crystal frequency [%d]\n"
	"  -d <device>               : serial device [%s]\n"
	"  -e {<sect>[..<sect>]|all} : erase FLASH sectors from..to, or all\n"
	"  -f <flashSize>            : FLASH size override\n"
	"  -g                        : debugging output (in honor of gcc's option -g)\n"
	"  -h <topic>                : display manual <topic>, start with \"contents\"\n"
	"  -i                        : print member info\n"
	"  -j {addr|LMA|LMA+1}       : jump to address [unset]\n"
	"  -k                        : read boot code version (ISP 'K' command)\n"
	"  -l <CRP level>            : lock code. Set code read protection (CRP) level [%d]\n"
	"  -m <RAM size>             : RAM 0 size override\n"
	"  -n                        : do not output controller as default action\n"
	"  -o <imageFile>            : set image output file (instead of stdout)\n"
	"  -p                        : probe baud rates and print to stdout. -q => only highest\n"
	"  -q                        : quick mode: prefer speed to security/kindness\n"
	"  -r <imageSize>            : read <imageSize> bytes from RAM/FLASH\n"
	"  -s {960@64>|file|flash}   : calculate CRC checksum from addr 64 to <1024 | whole file | whole FLASH\n"
	"  -t <time/ms>              : time "ISP" waits after RESET deassertion [%d]\n"
	"  -u                        : output UID\n"
	"  -v                        : verify FLASH contents after programming\n"
	"  -w                        : write to uC (default, if <inputFile> provided\n"
	"  -x                        : execute by issuing RESET without BSL\n"
	"  -y <sub-block size>       : binary transfer sub-block size [%d]\n"
	"  -z                        : zero output in case of success\n"
	"  -A                        : starting address of truly readable FLASH [%d]\n"
	"  -B 256,512,1024,4096      : define copy-ram-to-flash block sizes to be 256,512... \n"
	"  -C <CRP address>          : address of the 4-byte CRP code [0x%X]\n"
//	"  -D <deviceFile>           : use <deviceFile> for defining new devices.\n"
	"  -E                        : use echo on [normally off]\n"
	"  -F 4kix16,32kix14         : define FLASH sector layout to be 16x4kiB, then 14x32kiB\n"
//	"  -G                        :\n"
	"  -H                        * set error output handle [2][stderr]\n"
	"  -I <uC ID>                : set user-defined id\n"
//	"  -J                        :\n"
//	"  -K                        :\n"
	"  -L <CRP level>            : enable CRP level (dangerous especially for level 3) [%d]\n"
	"  -M 8ki@0x40000000,8ki@... : define uC RAM to be 8kiB at address 0x40000000 and ...\n"
	"  -N <uC number>            : set user-defined number\n"
//	"  -O                        * \n"
	"  -P {UUENCODE|BINARY}      : data encoding protocol override\n"
//	"  -Q                        :\n"
	"  -R <bottom>,<top>         : set RAM0 spaces reserved for ISP [0x%03X,0x%03X]\n"
	"  -S                        : show all compiled-in devices.\n"
	"  -T <time/ms>              : timeout of serial communication, step: 100ms [1000]\n"
	"  -U {<uC-number>|user}     : force uC like 2101, do not use part ID.\n"
	"  -V {CRC|READ}             * use verification method CRC or read & compare\n"
	"  -W <waveform>             : RESET/BSL control, see -h waveforms\n"
	"  -X <flow-control>         : execution flow control, see -h flow-control\n"
	"  -Y <flashBank>            : FLASH multiplexing: use flash bank <flashBank> for commands\n"
	"  -Z                        : zero output, even in case of failure. (return code only)\n"
	"\n"
	"<inputFile> can be in Intel hex-format (if named *.hex) or binary.\n\n"
	"If your controller does not work correctly with "ISP", read "ISP" -h troubleshooting\n"
	"Try -h contents for online manual\n"
	"\n"
	"Please report bugs to marc@windscooting.com\n"
	, address, baud, crystalHz, defaultDevice
	, levelCrpForce, timeAfterResetMs 
	, subBlockSize, remapBytes, addressCrp, levelCrpAllow
	, ispReserved[0], ispReserved[1]
	, timeoutSerialMs

	);
}

void manualContents(void) {
	printfOut(
	"The following topics are available from the online manual:\n"
	"  -h philosophy             : motivation behind "ISP"\n"
	"  -h numeric                : understanding "ISP"'s advanced command-line parsing\n"
	"  -h new-devices            : how to configure "ISP" for new devices\n"
	"  -h crp                    : code read protection (CRP)\n"
	"  -h flow-control           : debug output and control flow interruption\n"
	"  -h questionable           : operations considered questionable\n"
	"  -h waveforms              : defining /RST and /BSL signals\n"
	);
}

void manualQuestionable(void) {
	printfOut(
	"I (Marc, the author of this program) consider some operations of the LPC bootloader\n"
	"against common sense and in the context of this program these operations are called\n"
	"'questionable'. The questionable operations involve reading of sector 0. The manual(s)\n"
	"state that flash sector 0 cannot be read, because the boot ROM table is mapped into\n"
	"this location. So what? Isn't there a possibility to temporarily undo this mapping?\n"
	"However, it seems to me, that some devices implement common sense instead of user\n"
	"manual specifications, so I decided the user may select between options\n"
	"  -0 {<cmd>}+   : include sector 0 in commands, <cmd> may be:\n"
	"  S             : checksum calculation\n"
	"  I             : blank check\n"
	"  v             : blank check\n"
	"  l<n>          : define size of re-mapped area to be <n> bytes [%d]\n"
	"  .             : do nothing, except improve readability\n"
	"\n"
	"For blank-check, assume the sector is NOT blank. For verify, assume the sector to\n"
	"be correct.\n"
	, remapBytes
	);
}

void manualNumericArguments(void) {
	printfOut(
	"In the author's opinion, graphic user interfaces may be good thing for beginners\n"
	"but are always a nuisance for the advanced (scripting) user. However, we haven't\n"
	"seen much progress in command-line interfaces for years. The author hopes, that\n"
	ISP "'s advanced command line features will inspire other programmers to make\n"
	"their (command line) programs more user-friendly. "ISP" uses single char options\n"
	"that accept arguments that are either structured (lists, tuples) or have\n"
	"numeric extensions. Integers can be decimal, 0xhexadecimal, sometimes accept\n"
	"suffixes like k (*1000) or ki (*1024) which are suitable to define common\n"
	"memory sizes. A few examples:\n"
	"  -c 14746k        means: 14,746,000 (Hz) crystal frequency.\n"
	"  -x 0x40000000    means: start program at 0x4000 0000, which is the same as\n"
	"  -x 1Gi           (1GiB above 0).\n"
	"  -e 1             erases sector 1 (the range 1 to 1 of sectors).\n"
	"  -e 1..3          erases sectors 1 to 3.\n"
	"  -e all           erases all sectors of the device in use.\n"
	"  -B 256,512,1024,4096  defines the possible transfer sizes used to copy RAM\n"
	"                   to flash for most (all?) LPCs. It is a 4..5-tuple.\n"
	"Defining the flash sector sizes is done with a list of pairs, with a bit of\n"
	"eye-candy added, as an example LPC213x:\n"
	"  -F 4kix8,32kix14,4kix5      as the short form of\n"
	"  -F 4096x8,32768x14,4096x5\n"
	"Please note the difference between 4k (4000) and 4ki (4096).\n"
	"Also, don't expect every integer to have all formats allowed.\n"
	);
}

void manualNewDevices(void) {
	printfOut(
	"If any of the options -F, -M, -B, -N is provided, then ALL of them must be provided.\n"
	"This defines a device. -I should be provided to enable automatic selection.\n"
	"Simply put: these option allow you to extend the compiled in table to your needs.\n"
	"In the author's opinion, you should not need more than a datasheet of a new device\n"
	"to program it. No recompilation required. "ISP" uses RAM of your controller, that's\n"
	"why you have to define RAM location and size, too.\n"
	"Option -U user forces use of user-defined layout as does -N xxxx -U xxxx.\n"
	"Normally "ISP" reads the controller's device ID (part ID) and searches for a matching\n"
	"entry from the command line or from the compiled-in table. You have the option to\n"
	"specify an explicit device number (-U 2102 for a LPC2102) to make a hard selection\n"
	"in cases where the part ID is not useful or not wanted. Specifying -U user will\n"
	"always select your memory specification on the command line. Two convenience\n"
	"options are provided to specify total FLASH size (-f) and RAM0 size (-m) while still\n"
	"drawing most information out of the compiled-in table.\n"
	"Example: you want to program LPC2102. This controller has the same ID as LPC2103,\n"
	"yet the memory sizes are different.\n"
	"Option 1: define LPC2102 from scratch:\n"
	"  -F 4kix4 -M 4ki@1Gi -B 256,512,1024,4096 -N 2102 -U user\n"
	"Option 2: use compiled-in table, which yields LPC2103 and limit the memories:\n"
	"  -f 16ki -m 4ki\n"
	"Option 3: use compiled-in table, specify device number:\n"
	"  -U 2102\n"
	);
}

void manualCrp(void) {
	printfOut(
	ISP " takes care of CRP in the following sense: avoid unintended CRP and add CRP\n"
	"where needed. CRP3 or CRP4 (NO_ISP mode) are intentionally made a bit hard\n"
	"to achieve because "ISP" can put your controller into these CRP levels but\n"
	"not back.\n"
	"CRP levels are encoded at the memory location 0x%03X (4 bytes) - the CRP word.\n"
	ISP " allows upgrading CRP levels 0..2. This means, if your program image's\n"
	"CRP word is blank (0x%08X) or stands for CRP1 (0x%08X), CRP2\n"
	"(0x%08X), CRP3 (0x%08X), CRP4 (0x%08X) then you can choose between\n"
	"CRP0, CRP1, and CRP2 by command-line switches -l and -L.\n"
	"Or you can keep the image's CRP level 0..3 by providing -L only.\n"
	"Or you don't worry about CRP and thus avoid it by not providing\n"
	"neither -L nor -l. In short -l forces an exact CRP level while -L permits\n"
	"a range of CRP levels.\n"
	"If the CRP word contains a CRP pattern and CRP is not enabled (-L\n"
	"unspecified or 0) then an error is issued.\n"
	"\n"
	"CRP3 will only be executed if -l 3, -L 3 are given and the image contains\n"
	"the CRP word 0x%08X.\n"
	"This inhibits unintentional CRP3 if the program does not provide IAP.\n"
	"\n"
	"CRP4 will only be executed if -l 4, -L 4 are given and the image contains\n"
	"the CRP word 0x%08X.\n"
	"This inhibits unintentional CRP4 if the program does not provide IAP or\n"
	"your hardware does not provide SWD/JTAG.\n"
	"\n"
	, addressCrp
	, lpcCrpLevelToCode(0)
	, lpcCrpLevelToCode(1)
	, lpcCrpLevelToCode(2)
	, lpcCrpLevelToCode(3)
	, lpcCrpLevelToCode(4)
	, lpcCrpLevelToCode(3)
	, lpcCrpLevelToCode(4)
	);
}

void manualFlowControl(void) {
	printfOut(
	"Using "ISP"'s flow control mechanism, the user can insert delays or confirmation\n"
	"requests into the normal program flow. Rudimentary knowledge of the ISP protocol\n"
	"is required for this. Program flow can be interrupted/delays before every ISP command\n"
	"and in case of data transfer (ISP block sizes) before "ISP"'s sub-block-transfers.\n"
	"If ISP mode 'echo on' is used, any data write is followed by a data read, to which\n"
	"the 'cmd' r applies.\n"
	"  -X {<cmd>[<action>]|.}+   : execution flow control / prompt\n"
	"    where <cmd> may be one of the following (mostly) ISP command letters\n"
	"      '1'  : write of a single byte\n"
	"      'A'  : change echo\n"
	"      'B'  : change baud rate\n"
	"      'C'  : copy RAM to FLASH\n"
	"      'E'  : erase sector(s)\n"
	"      'G'  : go (execute code)\n"
	"      'I'  : blank check sectors\n"
	"      'J'  : read part ID\n"
	"      'K'  : read boot code version\n"
	"      'M'  : compare memory\n"
	"      'N'  : read part UID\n"
	"      'P'  : prepare sector(s) for writing\n"
	"      'R'  : read memory\n"
	"      'r'  : data block read memory/read echo line\n"
	"      'S'  : read CRC checksum\n"
	"      'U'  : unlock\n"
	"      'W'  : write to RAM\n"
	"      'w'  : data block of write to RAM\n"
	"\n"
	"    <action> can be:\n"
	"      omitted   : prompt user before proceeding. Program is stopped.\n"
	"      y         : prompt user, default to yes (continue)\n"
	"      n         : prompt user, default to no (quit program)\n"
	"      d<time>   : delay program by <time>\n"
	"\n"
	"    <time> can be:\n"
	"      <n>s      : <n> seconds.\n"
	"      <n>ms     : <n> milliseconds (may not be accurate).\n"
	"      <n>us     : <n> microseconds (may not be accurate).\n"
	"      <n>c      : <n> current baud rate charater timings.\n"
	"\n"
	"The user can react to the prompt by:\n"
	"  - simply hitting enter. This takes the default action.\n"
	"  - typing y or Y. This continues the program.\n"
	"  - typing n or N. This terminates the program.\n"
	"The character '.' can be used for improving readability - it has no effect.\n"
	"\n"
	"Depending on your operating system (some kind of *nix I hope) timings may be\n"
	"inaccurate. You need high resolution timer support for small delays of less\n"
	"than a few hundred milliseconds to be close to the value you provide.\n"
	"\n"
	"Example: -X.En.Wn.1d10us\n"
	"Prompt user before erasing or writing data to FLASH. Default action: terminate\n"
	"program.\n"
	"Wait 10us before sending a (write to RAM) data byte to the controller.\n"
	"\n"
	"Example: -XGy\n"
	"Prompt user before launching downloaded code. Default action: launch.\n"
	);
}

void manualPhilosophy(void) {
	printfOut(
	"The program "ISP" was originally designed because of the lack of well-behaved (*) ISP programmers\n"
	"for the NXP LPC21xx family.\n"
	"\n"
	"Well-behaved means: easily controllable by scripts/makefiles.\n"
	"\n"
	"Well-behaved means: reliability of operation (no blocking/deadlocking/need for killing it).\n"
	ISP " terminates after short time. Always. It does repond to SIGINT, although you don't need that.\n"
	"\n"
	"Well-behaved means: provide options to influence the program but don't require the user to use\n"
	"these options on every invokation. Which leads us directly to the next point:\n"
	"\n"
	"Well-behaved means: simple invokation and reasonable defaults for the options.\n"
	"\n"
	"Well-behaved means: provide functionality, not policy.\n"
	"You can use "ISP" for retrieving information from the microcontroller and use that information in\n"
	"a script. For example you can use "ISP" to read the UID and output the UID (and only the UID and\n"
	"and nothing but the UID) to stdout. You can write a certain sector of the controller with some\n"
	"data important to you - it does not have to be a program at all.\n"
	"\n"
	"Well-behaved means: it is NOT a GUI program, asshole, if you didn't get it until here!! If you\n"
	"plan to provide a user interface to this program - a pretty simple task, because of "ISP"'s\n"
	"script-oriented design (funny, isn't it?) - please DON'T DO it. Mentally disabled persons should\n"
	"not do programming at all. Don't let them multiply their bullshit. Please move on and reduce the\n"
	"functionality of another program, thank you very much.\n"
	"\n\n"
	"(*) from the author's point of view.\n"
	);
}

void manualTroubleshooting(void) {
	printfOut(
	ISP " may not work immediately with your controller. Different controllers support\n"
	"different communication speeds and different ISP command sets and maybe bugs.\n"
	"The USB-to-serial converters commonly used nowadays sometimes do not provide\n"
	"the same level of control as the original PC UARTs did. This can lead to\n"
	"unexpected timing/control line behaviour.\n"
	ISP" by its design does NOT support flow control, i.e. won't lower speed when the\n"
	"controller asks to. This is especially important in 'write to ram' commands. I\n"
	"experienced few problems because of this lacking feature and it can be fixed by\n"
	"inserting delays, see below.\n"
	"I recommend the following action, if your controller does not work with "ISP"'s\n"
	"default settings:\n"
	"  - if your controller does not have an IRC, set the correct crystal frequency.\n"
	"    "ISP"'s default is %dHz\n"
	"  - if your controller is not responding, try baud probing (-p)\n"
	"  - if your controller is not responding, try lower baud rate, different reset\n"
	"    time (-t option, depends on the board of your controller) or different\n"
	"    serial timeout (-T option). Be careful: increasing these times increases\n"
	"    time required for auto-probing.\n"
	" - if ISP failes while downloading, try option -E. This option enables command\n"
	"   echoing by the controller and does some flow control as a side effect.\n"
	" - if ISP failes while downloading, try option -X1d1c. This option inserts\n"
	"   a delay of 1 char after each char sent. This effectively slows down the\n"
	"   data rate, the controller has to handle. You may use numbers greater than\n"
	"   1 or a different delay specification at your discretion. Your system\n"
	"   should have high resolution timer (kernel feature) support for this.\n"
	" - verify the RAM usage of the ISP handler of your controller. Option -R.\n"
	"   Overwriting the ISP-handler's data area of stack can result in really\n"
	"   everything you can imagine or not.\n"
	" - verify the protocol used for data transfers. The LPC800 series introduced\n"
	"   a binary protocol, not seen before on other LPC devices. option -P.\n"
	, crystalHz
	);
}

void manualWaveforms(void) {
	printfOut(
	"Depending on your board and communication devices, you need to adapt the\n"
	"sequence, timing and levels of RS-232 signals RTS and DTR.\n"
	"Typically DTR connects (after level shifting) to LPC's /RST and\n"
	"RTS connects (after level shifting) to /BSL. Both signals are active low\n"
	"at the LPC part. The -W option lets you define, how to apply RTS and DTR to\n"
	"your specific board to get it reset and/or to start the ISP handler.\n"
	"To define a sequenc of signals with timing, you have to write a sequence of\n"
	"the following symbols:\n"
	"  D    : set DTR (normally /RST) to logic HIGH (1)\n"
	"  d    : set DTR (normally /RST) to logic LOW (0)\n"
	"  R    : set RTS (normally /BSL) to logic HIGH (1)\n"
	"  r    : set RTS (normally /BSL) to logic LOW (0)\n"
	"  P    : long pause; wait for the time specified in -t option\n"
	"  p    : short pause, about 0.1s (enough for (de-)activating /BSL\n"
	"         and in most systems enough for activating /RST).\n"
	"  y    : user prompt, pause until user hits <ENTER>\n"
	"  i=   : sequence: enter ISP\n"
	"  x=   : sequence: execute program using RESET\n"
	"  j=   : sequence: terminate "ISP" without resetting for execute by jump\n"
	"  e=   : sequence: terminate "ISP" with error - hold RESET active\n"
//	"  p=<time>\n"
	"  .    : fill character for improving readability (same is true for '=')\n"
	"\n"
	"Providing option -W empties all sequences before parsing. Every sequence\n"
	"you do not define will be empty and do nothing at all.\n"
	"\n"
	"Let's have a look at the default definitions, to understand the syntax:\n"
	"  -W %s\n"
	"i: This enables entry to the ISP handler, by setting /BSL and /RST to 0,\n"
	"then, after a delay /RST to 1 and after a further delay /BSL to 1 and\n"
	"starts synchronizing.\n"
	"x: This sets /RST to 0, waits, then sets /RST to 1.\n"
	"j: keeping signals as they are (all 1) is OK.\n"
	"e: This sets /RST to 0 and waits for /RST to take effect.\n"
	"\n"
	"Suppose, you have a board that doesn't have /BSL connected to the serial\n"
	"connector but to a pushbutton. /RST is connected to /RTS, but inverted.\n"
	"For such a board you can use the following waveforms:\n"
	"  -W i=RyrP.x=RPr.j=.e=RP\n"
	"i: This enables entry to the ISP handler by setting RST to 1, then promting\n"
	"the (slow) user. User presses and holds the BSL-button, then hits <ENTER>.\n"
	ISP" proceeds by setting RST to 0, waiting and finally synchronizing.\n"
	, defaultW
	);
}

/** Changes the code read protection word, depending of an allowed maximum CRP level and a desired CRP level.
 * @param image the FLASH image contents.
 * @param addressCrp the address of the word encoding the CRP level, typically 0x2FC.
 * @param address the destination base address of the image.
 * @param crpMax maximum (allowed) CRP level. Must be an exect match for negative levels.
 * @param crpDesired the desired CRP level. -1 if desired level is to be considered unset.
 */
bool fifoLpcCrp(Fifo *image, Uint32 addressCrp, Uint32 address, int crpMax, int crpDesired) {

	if (addressCrp&3 || address&3) {
		printfError("ERROR: Alignment error of destination address or CRP bytes.\n");
		return false;
	}
	if (crpMax>4 || crpMax<0) {
		printfError("ERROR: Maximum CRP level (-L) out of range: %d. Allowed: 0..4 .\n",crpMax);
		return false;
	}
	if (crpDesired>4 || crpDesired<-1) {
		printfError("ERROR: CRP level (-l) out of range: %d. Allowed: 0..4 .\n", crpDesired);
		return false;
	}
	if (!lpcCrpLevelAllowed(crpMax,crpDesired)) {
		printfError("ERROR: CRP level (-l) not allowed. Allow with option -L%d\n",crpDesired);
		return false;
	}
	if (addressCrp<address) {
		if (crpDesired>-1) {	// explicit set
			printfError("ERROR: cannot provide CRP: destination address 0x%08X too high.\n",address);
			return false;
		}
		else return true; 	// nothing to do
	}

	Fifo clone = *image;
	if (fifoCanRead(&clone)>addressCrp-address) {	// don't worry: there is no partial write of the CRP word
		fifoSkipRead(&clone,addressCrp-address);
		Fifo writer;
		fifoInitRewrite(&writer,&clone);
		Uint32 crpWord = 0;
		if (!fifoGetUint32(&clone,&crpWord)) return false;
		const int crpExe = lpcCrpCodeToLevel(crpWord);
		if (crpExe>crpMax) {
			printfWarning("WARNING: image has CRP (level %d)\n",crpExe);
		}

		// checking formal conditions
		const int severeLevels[] = { 3,4 };
		for (int i=0; i<ELEMENTS(severeLevels); i++) {
			const int crpSev = severeLevels[i];
			if ((crpDesired==crpSev || crpMax==crpSev /*|| crpExe==crpSev*/)	// attempt for CRP3,4
			&& !(crpDesired==crpSev && crpMax==crpSev && crpExe==crpSev)) {	// requirement for CRP3,4 
				printfError("ERROR: CRP level %d needs -l%d -L%d and image to contain 0x%08X@0x%03X.\n",
					crpSev,crpSev,crpSev, lpcCrpLevelToCode(crpSev),addressCrp);
				return false;
			}
		}

		// keep image's CRP level, if allowed
		if (crpDesired==-1 && crpExe>crpMax) {
			printfError("ERROR: image CRP (level %d) not allowed.\n",crpExe);
			return false;
		}
		// setting crp level
		if (crpDesired!=-1	// forced CRP level
		&& crpExe!=crpDesired) {
			if (crpWord==~0u	// erased memory pattern
			|| crpExe>0) {		// any CRP level
				printfWarning("WARNING: image location 0x%04X (CRP) modified.\n",addressCrp);
				fifoPutInt32(&writer,lpcCrpLevelToCode(crpDesired));
			}
			else {
				printfError("ERROR: image location 0x%04X (CRP) must be blank (-1)"
						" or CRP level 1..4\n",addressCrp);
				return false;
			}
		}
		return true;
	}
	else {	// CRP word not part of the image
		if (crpDesired<=0) return true;	// we don't try, what we can't
		else {
			printfError("ERROR: image too small for CRP.\n");
			return false;
		}
	}
}

bool lpcCrp(const HexImage *hexImage, Uint32 addressCrp, int crpMax, int crp) {
	Fifo fifo;
	fifoInitRead(&fifo,(char*)hexImage->segments[0].data, hexImage->segments[0].size);
	return fifoLpcCrp(&fifo,addressCrp,hexImage->segments[0].address,crpMax,crp);
}

int noConnection(void) {
	printfError("ERROR: no connection - try different baud rate and especially crystal frequency?\n");
	return 1;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Flow control hook
//
enum FlowControlAction {
	FC_NO_ACTION,		///< no action
	FC_DELAY,		///< unconditional delay
	FC_WAIT_YES,		///< wait for user input
	FC_WAIT_NO,		///< wait for user input
};

typedef struct {
	int		command;	///< most often, the ISP command letter.
	int		action;
	int		delayUs;	///< negative values indicate character units
} FlowControl;

static FlowControl flowControls[] = {
	{ '1' },
	{ 'A' },
	{ 'B' },
	{ 'C' },
	{ 'E' },
	{ 'G' },
	{ 'I' },
	{ 'J' },
	{ 'K' },
	{ 'M' },
	{ 'N' },
	{ 'P' },
	{ 'R' },
	{ 'S' },
	{ 's' },
	{ 'U' },
	{ 'W' },
	{ 'w' },
};

Int64 baudBitsToUs(int baudBits) {
	return 1000ll*1000*baudBits/baud;
}

FlowControl* flowControlFind(int command) {
	for (int i=0; i<ELEMENTS(flowControls); i++)
		if (command==flowControls[i].command) return &flowControls[i];
	return 0;
}

bool flowControlConfig(int command, int action, int delayUs) {
	FlowControl *fc = flowControlFind(command);
	if (fc!=0) {
		fc->action = action;
		fc->delayUs = delayUs;
		return true;
	}
	else return false;
}

static void safeUsleep(long long us) {
	if (us<1000*1000) usleep((int)us);
	else {
		struct timespec ts = {
			.tv_sec = us / 1000000,
			.tv_nsec = (us % 1000000) * 1000
		};
		nanosleep(&ts,0);
	}
}

bool flowControlHook(char command, const char* format, ...) {
	FlowControl *fc = flowControlFind(command);
	if (fc==0 || fc->action==FC_NO_ACTION) return true;
	else {
		switch(fc->action) {
			case FC_DELAY:
				if (fc->delayUs>=0) safeUsleep(fc->delayUs);
				else safeUsleep(baudBitsToUs(-fc->delayUs*10/*bits per char*/));
				return true;
			case FC_WAIT_YES:
			case FC_WAIT_NO: {
				const bool yes = fc->action==FC_WAIT_YES;
				printfNormal("Stopped at: ");
				va_list args;
				va_start(args,format);
				vprintfNormal(format,args);
				va_end(args);
				printfNormal(" - Continue %s", yes ? "([Y]/n) ? " : "(y/[N]) ? ");
				char line[100];
				const int n = read(0,line,sizeof line -1);
				if (n>=1) {
					line[n] = 0;
				}
				else {
					printfError("ERROR: could not read user input.\n");
					return false;
				}
				switch(line[0]) {
					case 0:
					case '\n':
					case '\r':	if (!yes) printfNormal("Operation canceled by user.\n");
							return yes;	// the default
					case 'y':
					case 'Y':	return true;
					case 'n':
					case 'N':	printfNormal("Operation canceled by user.\n");
							return false;
					default :	printfError("ERROR: invalid user input.\n");
							return false;
				}
			}
			default: 
				printfError("ERROR: invalid control flow action.\n");
				return false;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// RTS/DTR wave form definitions.
//
enum {
	WAVE_ISP,		///< wave form to get into ISP handler
	WAVE_EXECUTE,		///< wave form to RESET and start FLASH program
	WAVE_JUMP,		///< wave form to be able to start FLASH program by a jump
	WAVE_ERROR,		///< wave form to keep RESET on device
	WAVE_COUNT		///< number of wave forms
};

enum {
	WAVE_CMD_END,
	WAVE_CMD_D0,
	WAVE_CMD_D1,
	WAVE_CMD_R0,
	WAVE_CMD_R1,
	WAVE_CMD_PAUSE_SHORT,
	WAVE_CMD_PAUSE_LONG,
	WAVE_CMD_PROMPT,

	WAVE_CMDS_MAX = 10,
};

typedef char WaveCommand;
typedef WaveCommand Wave[WAVE_CMDS_MAX+1];	// always terminating 0 required

static Wave waves[WAVE_COUNT] = {};
static const char* const wavePrompts[WAVE_COUNT] = {
	"ISP entry (Synchronize)",
	"RESET and start uC program",
	"Start uC program by branch instruction",
	"Apply RESET and keep it"
};

typedef struct {
	int	pauseShortMs;
	int	pauseLongMs;
} WaveConfiguration;

bool wavePrompt(const char* format, ...) {
	va_list args;
	va_start(args,format);
	vprintfNormal(format,args);
	va_end(args);
	printfNormal(" - press <ENTER> to continue.");
	char line[100];
	const int n = read(0,line,sizeof line -1);
	return true;
}

bool wavePlay(int fd, WaveConfiguration const *conf, int no) {
	if (0<=no && no<WAVE_COUNT) {
		const WaveCommand *wave = waves[no];

		while (*wave!=WAVE_CMD_END) {
			switch(*wave) {
			case WAVE_CMD_D0: serialSetDtr(fd,false); break;
			case WAVE_CMD_D1: serialSetDtr(fd,true); break;
			case WAVE_CMD_R0: serialSetRts(fd,false); break;
			case WAVE_CMD_R1: serialSetRts(fd,true); break;
			case WAVE_CMD_PAUSE_SHORT: safeUsleep(conf->pauseShortMs*1000); break;
			case WAVE_CMD_PAUSE_LONG: safeUsleep(conf->pauseLongMs*1000); break;
			case WAVE_CMD_PROMPT: wavePrompt(wavePrompts[no]); break;
			default:
				printfError("INTERNAL ERROR: wavePlay() #1.\n");
			}
			wave++;
		}
		return true;
	}
	else {
		printfError("INTERNAL ERROR: wavePlay() #2.\n");
		return false;
	}
}

/** Translates the input of the -W option into waves data structure.
 */
bool waveCompile(Fifo* input) {
	Fifo clone = *input;
	memset(waves,0,sizeof waves);
	int no=-1;
	int seq = 0;
	while (fifoCanRead(&clone)) {
		const char c = fifoRead(&clone);
		switch(c) {	// handle wave selection
			case 'i': no = WAVE_ISP;	seq = 0; continue;
			case 'x': no = WAVE_EXECUTE;	seq = 0; continue;
			case 'j': no = WAVE_JUMP;	seq = 0; continue;
			case 'e': no = WAVE_ERROR;	seq = 0; continue;
			case '=':
			case '.': continue;
		}

		// now comes a command (or error)

		// make sure, the wave number is selected already.
		if (no==-1) {
			printfError("ERROR: parsing -W: missing i,x,j or e.\n");
			return false;
		}

		// make sure there's space for the command
		if (seq<WAVE_CMDS_MAX) ; // fine: we can put in another and still not lose the final 0
		else {
			printfError("ERROR: too many commands in single waveform definition, maximum is %d\n",
				WAVE_CMDS_MAX);
			return false;
		}

		// store the command
		switch(c) {
			case 'D': waves[no][seq++] = WAVE_CMD_D1; break;
			case 'd': waves[no][seq++] = WAVE_CMD_D0; break;
			case 'R': waves[no][seq++] = WAVE_CMD_R1; break;
			case 'r': waves[no][seq++] = WAVE_CMD_R0; break;
			case 'y': waves[no][seq++] = WAVE_CMD_PROMPT; break;
			case 'p': waves[no][seq++] = WAVE_CMD_PAUSE_SHORT; break;
			case 'P': waves[no][seq++] = WAVE_CMD_PAUSE_LONG; break;
			default:
				printfError("ERROR: invalid character '%c' in -W waveform definition.\n",c);
				return false;
		}
	}

	fifoCopyReadPosition(input,&clone);
	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Probing

bool trySerialParams(const char *device, Uint32 baud, const WaveConfiguration *wc, Uint32 timeoutSerialMs, Uint32 crystalHz) {
		printfDebug("Probing: "ISP" -d%s, -b%d -t%d -T%d\n",device,baud,wc->pauseLongMs,timeoutSerialMs);
		int fd = serialOpenBlockingTimeout(device,baud,timeoutSerialMs/100);

		wavePlay(fd,wc,WAVE_ISP);
		const bool result = lpcSync(fd,crystalHz);
		wavePlay(fd,wc,WAVE_JUMP);	// control back to ISP
		close(fd);
		return result;
}

int probeBauds(const char *device, const WaveConfiguration *wc, Uint32 timeoutSerialMs, Uint32 crystalHz, bool quick) {

	int highest = 0;
	Uint32 bauds[] = { 230400, 115200, 57600, 38400, 19200, 9600 };
	for (int i=0; i<ELEMENTS(bauds); ++i) {
		if (trySerialParams(device,bauds[i],wc,timeoutSerialMs,crystalHz)) {
			highest = int32Max(highest,bauds[i]);
			printfOut(ISP " -d%s, -b%d -t%d -T%d\n",device,bauds[i],wc->pauseLongMs,timeoutSerialMs);
			if (quick) break;
		}
	}
	return highest;
}


bool startsWith(const char *prefix, const char *string) {
	while (*prefix) {
		if (*prefix!=*string) return false;
		prefix++; string++;
	}
	return true;
}

int main(int argc, const char **argv) {
	struct timespec ts;
	clock_getres(CLOCK_REALTIME,&ts);
	printfDebug("Clock resolution: %lds %ldns\n",ts.tv_sec,ts.tv_nsec);

	char lineBuffer[1024];
	Fifo fifoCmdLine = { lineBuffer, sizeof lineBuffer };
	char accumulateBuffer[1024];
	Fifo fifoNonOptions = { accumulateBuffer, sizeof accumulateBuffer };	// buffer sharing NOT possible here!

	const char * const environmentVariable = "MXLI_PARAMETERS";
	const char * const environmentParameters = getenv(environmentVariable);
	if (environmentParameters!=0) {
		const int n = strlen(environmentParameters);
		ReadFifo fifoE = { (char*)environmentParameters, .size = n, .wTotal = n, };
		Fifo word;
		while (fifoParseUntil(&fifoE,&word," ")		// parse space-separated word
		|| (fifoParseStringNonEmpty(&fifoE,&word)) ) {	// or last word = rest of string
			if (fifoPutFifo(&fifoCmdLine,&word)
			&& fifoqPrintNext(&fifoCmdLine)) {
				// fine
			}
			else {
				printfError("ERROR: command line too long .\n");
				goto failEarly;
			}
		}
	}

	if (!fifoPoptAccumulateCommandLine(&fifoCmdLine,argc,argv)) {
		printfError("ERROR: command line too long.\n");
		goto failEarly;
	}

	if (fifoPoptScanOption(&fifoCmdLine,'?',"help")) {
		showHelp();
		goto returnEarly;
	}

	bool minus = false;
	while (fifoCanRead(&fifoCmdLine)) {
		if (false
		|| fifoPoptBool(&fifoCmdLine,optionBools,&minus)
		|| fifoPoptInt32Flag(&fifoCmdLine,optionIntFlags,&minus)
		|| fifoPoptInt32(&fifoCmdLine,optionInts,&minus)
		|| fifoPoptInt32Range(&fifoCmdLine,optionInt32Ranges,&minus)
		|| fifoPoptInt32Tuple(&fifoCmdLine,optionInt32Tuples,&minus)
		|| fifoPoptString(&fifoCmdLine,optionStrings,&minus)
		|| fifoPoptSymbol(&fifoCmdLine,optionSymbols,&minus)
		|| fifoPoptNonOptionAccumulate(&fifoCmdLine,&fifoNonOptions,&minus)
		) {
			// fine
		}
		else {
			printfError("ERROR: invalid option/parameter: \"%s\"\n",fifoReadPositionToString(&fifoCmdLine));
			printfError("Try \"-?\" for help\nTry \"-h contents\" for manual overview\n");
			goto failEarly;
		}
	}

	// we could not output that earlier, because -g was not active!
	if (environmentParameters!=0) printfDebug("Using parameters of %s=%s\n",environmentVariable,environmentParameters); 

	if (manualSymbol>-1) {
		switch(manualSymbol) {
			case 0: manualContents(); goto returnEarly;
			case 1: manualNumericArguments(); goto returnEarly;
			case 2: manualNewDevices(); goto returnEarly;
			case 3: manualCrp(); goto returnEarly;
			case 4: manualFlowControl(); goto returnEarly;
			case 5: manualQuestionable(); goto returnEarly;
			case 6: manualPhilosophy(); goto returnEarly;
			case 7: manualTroubleshooting(); goto returnEarly;
			case 8: manualWaveforms(); goto returnEarly;
			default: printfError("ERROR: internal manual error\n"); goto failEarly;
		}
	}

	// early parsing

	// formal checking
	if (0<=commandJAnswerNumbers && commandJAnswerNumbers<=2) ; 	// OK.
	else {
		printfError("ERROR: invalid number of device ID words: %d. Allowed: 0..2\n",commandJAnswerNumbers);
		goto failEarly;
	}

	int nFs = 0;
	int nMs = 0;
	if (fifoCanRead(&fifoM)) {
		if (1<=(nMs=fifoParseListPairInt(&fifoM,&parseM,listM,ELEMENTS(listM)))) {
			printfDebug("Explicit memory definition: %dkiB RAM at 0x%08X.\n",listM[0].a/1024,listM[0].b);
		}
		else {
			printfError("ERROR: invalid explicit memory definition.\n");
			goto failEarly;
		}
	}
	if (fifoCanRead(&fifoF)) {
		if (1<=(nFs=fifoParseListPairInt(&fifoF,&parseF,listF,ELEMENTS(listF)))) {
			printfDebug("Explicit sector layout definition:");
			for (int i=0; i<nFs; ++i) {
				if (i!=0) fprintf(stderr,", ");
				printfDebug("%d sectors of %dkiB",listF[i].b,listF[i].a/1024);
			}
			printfDebug("\n");
		}
		else {
			printfError("ERROR: invalid explicit sector layout definition.\n");
			goto failEarly;
		}
	}
	while (fifoCanRead(&fifoX)) {
		const char cmd = fifoRead(&fifoX);
		if (cmd=='.') continue;	// . as optional separator
		FlowControl *fc = flowControlFind(cmd);
		if (fc!=0) {	// valid command
			if (fifoParseExactChar(&fifoX,'y')) {
				if (!flowControlConfig(cmd,FC_WAIT_YES,0)) goto failEarly;
			}
			else if (fifoParseExactChar(&fifoX,'n')) {
				if (!flowControlConfig(cmd,FC_WAIT_NO,0)) goto failEarly;
			}
			else if (fifoParseExactChar(&fifoX,'d')) {
				int n;
				int power;
				if (fifoParseInt(&fifoX,&n)) {
					if (fifoParseTechnicalUnit(&fifoX,&power,"s")) {
						switch(power) {
						case 0: if (!flowControlConfig(cmd,FC_DELAY,n*1000*1000)) goto failEarly;
							break;
						case -3: if (!flowControlConfig(cmd,FC_DELAY,n*1000)) goto failEarly;
							break;
						case -6: if (!flowControlConfig(cmd,FC_DELAY,n)) goto failEarly;
							break;
						default:	printfError("ERROR: value out of range/resolution.\n");
								goto failEarly;
						}
					}
					else if (fifoParseTechnicalUnit(&fifoX,&power,"c")) {
						switch(power) {
						case 0: if (!flowControlConfig(cmd,FC_DELAY,-n)) goto failEarly;
							break;
						case 3: if (!flowControlConfig(cmd,FC_DELAY,-n*1000)) goto failEarly;
							break;
						case 6: if (!flowControlConfig(cmd,FC_DELAY,-n*1000*1000)) goto failEarly;
							break;
						default:	printfError("ERROR: value out of range/resolution.\n");
								goto failEarly;
						}
					}
					else {
						printfError("ERROR: invalid unit.\n");
						goto failEarly;
					}
				}
				else {
					printfError("ERROR: delay expected.\n");
				}
			}
		}
		else {
			printfError("ERROR: flow control on command '%c' not implemented.\n",cmd);
			goto failEarly;
		}
	}
	// waveform definition
	if (fifoCanRead(&fifoW)) {	// -W specified
		printfDebug("Using user-supplied waveform.\n");
		if (!waveCompile(&fifoW)) goto failEarly;
	}
	else {	// use default definition
		printfDebug("Using default waveforms.\n");
		ReadFifo fifoDefW = { defaultW, .wTotal = sizeof defaultW-1, .size = sizeof defaultW-1 };
		if (!waveCompile(&fifoDefW)) goto failEarly;
	}

	bool doBlankCheck0 = false;
	bool doVerifyLower = false;
	bool doCrcLower = false;
	while (fifoCanRead(&fifo0)) {
		const char cmd = fifoRead(&fifo0);
		switch(cmd) {
			case 'I':	doBlankCheck0 = true; break;
			case 'v':	doVerifyLower = true; break;
			case 'S':	doCrcLower = true; break;
			case 'l':	if (fifoParseInt(&fifo0,&remapBytes)) break;
					else {
						printfError("ERROR: invalid parameter for option -0l: \"");
						printfErrorFifo(fifo0);
						printfError("\"\n");
						goto failEarly;
					}
			case '.':	break;	// NOP
			default:	printfError("ERROR: invalid suboption to -0: \"");
					printfErrorFifo(fifo0);
					printfError("\"\n");
					goto failEarly;
		}
	}

	// check, if execution specification is not contradictory:
	if (executeReset) {	// RESET method specified
		if (executeSymbol!=-1 || executeAddress!=-1) {
			printfError("ERROR: specifying both -x and -j is contradictory.\n");
			goto failEarly;
		}
	}

	// Explicit file on command line is intended for writing as default.
	const char *inputFile = fifoCanRead(&fifoNonOptions) ? fifoReadPositionToString(&fifoNonOptions) : 0;
	if (inputFile) writeAll = true;

	const char *device = fifoCanRead(&fifoDevice) ? fifoReadPositionToString(&fifoDevice) : defaultDevice;
	
	const LpcFamilyMember * const * const lpcFamilyMembers = lpcFamilyMembersXxxx;	// used in this program

	// First command: list compiled-in devices
	if (showAllDevices) {
		printfOut("Compiled-in devices:\n");
		for (int d=0; lpcFamilyMembers[d]!=0 ; d++) {
			const LpcFamilyMember *m = lpcFamilyMembers[d];
			printfOut("  ID 0x%08X : %s\n",m->id,m->name);
		}
	}

	const WaveConfiguration waveConfiguration = {
		.pauseShortMs = 100,
		.pauseLongMs = timeAfterResetMs,
	};

	// Command:
	if (probeBaud) {
		baud = probeBauds(device,&waveConfiguration,timeoutSerialMs,crystalHz,quick);
		if (baud==0) goto failEarly;
	}

	printfDebug("Communication device=%s, speed=%dbd, uC crystal=%dkHz\n",device,baud,crystalHz/1000);
	int fd = serialOpenBlockingTimeout(device,baud,timeoutSerialMs/100);

	if (!wavePlay(fd,&waveConfiguration,WAVE_ISP)) goto failErrorClose;

	Uint32 partIds[2] = { 0,0 };
	if (lpcSync(fd,crystalHz)
	&& lpcBaud(fd,baud,1)
	&& lpcEcho(fd,&fifoIn,useEcho)
	&& lpcReadPartId(fd,&fifoIn,partIds)
	) ;	// fine
	else return noConnection();

	// calculate device
	// user (command-line) defined device...
	LpcFamilyMemoryMap familyMemoryMapUser = {};
	LpcFamilyMember familyMemberUser = {
		.familyMemoryMap = &familyMemoryMapUser
	};

	if (nFs>=1		// option -F
	|| nMs>=1		// option -M
	|| listB[0]!=0		// option -B
	|| fifoCanRead(&fifoN)	// option -N
	|| userId!=-1		// option -I
	) if (nFs>=1 && nMs>=1 && listB[0]!=0 && fifoCanRead(&fifoN)) {
		if (nFs<1) {
			printfError("Invalid FLASH layout definition.\n");
			goto failErrorClose;
		}
		familyMemoryMapUser.addressFlash = 0;
		for (int ss=0; ss<nFs; ++ss) {
			familyMemoryMapUser.sectorArrays[ss].sizeK = listF[ss].a / 1024;
			familyMemoryMapUser.sectorArrays[ss].n = listF[ss].b;
		}

		if (nMs<1) {
			printfError("Invalid RAM layout definition.\n");
			goto failErrorClose;
		}
		for (int ms=0; ms<nMs; ++ms) {
			familyMemberUser.sizeRamKs[ms] = listM[ms].a / 1024;
			familyMemoryMapUser.addressRams[ms] = listM[ms].b;
		}

		for (int b=0; b<ELEMENTS(familyMemoryMapUser.blockSizes); ++b)
			familyMemoryMapUser.blockSizes[b] = listB[b];

		familyMemberUser.sizeFlashK = lpcFamilySectorOffset(&familyMemoryMapUser,
			lpcFamilySectorHighest(&familyMemoryMapUser)+1) / 1024;
		familyMemberUser.name = fifoCanRead(&fifoN) ? fifoReadLinear(&fifoN) : "LPC 0";
		familyMemberUser.id = userId;		// -1 hopefully matches no real device. 0 neither.
		//userDefinedAvailable = true;
	}
	else {
		printfError("All of the options -F, -M, -B, -N are required for defining new device.\n");
		goto failErrorClose;
	}

	// select device: user/probed by id/number
	if (fifoCanRead(&fifoU)) manuallySelected = fifoReadLinear(&fifoU);	// uC selected by -U

	// the user-defined device is always checked first, to be able to override table entries
	LpcFamilyMember member = {};
	LpcFamilyMemoryMap memoryMap = {};
	bool deviceDetected = true;
	if (ucSymbol!=-1 && manuallySelected!=0) {
		printfError("Competing options -U.\n");
		goto failErrorClose;
	}
	
	if (ucSymbol==0) {			// 'user' selected, the user's command line-defined device.
		member = familyMemberUser;
		memoryMap = familyMemoryMapUser;
		deviceDetected = false;
	}
	else if (manuallySelected) {
		LpcFamilyMember const *tMember = lpcFindByName(lpcFamilyMembers,manuallySelected);
		if (tMember!=0) {
			member = *tMember;
			memoryMap = *tMember->familyMemoryMap;
			deviceDetected = false;
		}
		else {
			printfError("Cannot find %s\n",manuallySelected);
			goto failErrorClose;
		}
	}
	else {	// find by ID
		if (partIds[0]==familyMemberUser.id) {
			member = familyMemberUser;
			memoryMap = familyMemoryMapUser;
		}
		else {
			LpcFamilyMember const *tMember = lpcFindById(lpcFamilyMembers,partIds[0]);
			if (tMember!=0) {
				member = *tMember;
				memoryMap = *tMember->familyMemoryMap;
			}
			else {
				printfError("Cannot find part ID0x%08X",partIds[0]);
				goto failErrorClose;
			}
		}
	}
	member.familyMemoryMap = &memoryMap;	// pointer rearrangement required.
	printfDebug("%s: %s, FLASH:%dkiB, RAM0:%dkiB, RAM1:%dkiB, RAM2:%dkiB\n",
		deviceDetected ? "Detected" : "Selected",
		member.name,member.sizeFlashK,member.sizeRamKs[0],member.sizeRamKs[1],member.sizeRamKs[2]
	);

	if (!noUcOutput && deviceDetected) {
		printfNormal("Detected: %s, FLASH:%dkiB, RAM0:%dkiB, RAM1:%dkiB, RAM2:%dkiB\n",
			member.name,member.sizeFlashK,member.sizeRamKs[0],member.sizeRamKs[1],member.sizeRamKs[2]
		);
	}

	// now patch the device with command line overrides
	if (flashSizeOverride!=-1) member.sizeFlashK = flashSizeOverride/1024;
	if (memorySizeOverride!=-1) member.sizeRamKs[0] = memorySizeOverride/1024;
	if (ispProtocolOverride!=-1) member.ispProtocol = ispProtocolOverride;

	// Command: show info, if desired
	if (showInfo) {
		char infoBuffer[2048];
		Fifo fifoInfo = { infoBuffer, sizeof infoBuffer };
		fifoPrintLpcFamilyMember(&fifoInfo,&member);
		printfOut("Device information:\n");
		printFifo(stdout,&fifoInfo);
		printfOut("\n");
	}

	if (showBoot) {
		Uint32 bootCodeVersion = 0;
		if (lpcReadBootCodeVersion(fd,&fifoIn,&bootCodeVersion)) {
			printfOut("%d.%d\n",bootCodeVersion>>8,bootCodeVersion&0xff);
		}
		else goto failErrorClose;
	}

	if (showUid) {
		Uint32 uid[4];
		if (lpcReadUid(fd,&fifoIn,uid)) {
			printfOut("0x%08X 0x%08X 0x%08X 0x%08X\n",uid[0],uid[1],uid[2],uid[3]);
		}
		else {
			printfError("ERROR: cannot read UID.\n");
			goto failErrorClose;
		}
	}

	// early check of set flashbank argument
	if (setFlashBankSymbol!=-1) {
		if (setFlashBank==-1) {
			// there is only one symbol so far.
			setFlashBank = fBank;
		}
		else {
			printfError("ERROR: contradicting flash bank options.\n");
			goto failErrorClose;
		}
	}

	const LpcProgramConfig lpcProgramConfig = {
		.verify = verify,
		.fast = quick,
		.ramLowerSpace = ispReserved[0],
		.ramUpperSpace = ispReserved[1],
		.doBlankCheck0 = doBlankCheck0,
		.doVerifyLower = doVerifyLower,
		.doCrcLower = doCrcLower,
		.remapBytes = remapBytes,
		.fBank = fBank,
	};

	if (blankCheck) {
		const int sectorFrom = 0;
		const int sectorTo = lpcAddressToSector(
			&member, member.familyMemoryMap->addressFlash+member.sizeFlashK*1024 -1);
		bool blank;
		if (quick) {	// fast mode
			printfOut("Blank check sector %d..%d: ",sectorFrom,sectorTo);
			if (lpcBlankCheck(fd,&fifoIn,sectorFrom,sectorTo,fBank,&blank,doBlankCheck0)) {
				printfOut("%s\n",blank ? "OK" : "FAILED");
			}
			else goto failErrorClose;
		}
		else {	// detailed mode
			for (int s=sectorFrom; s<=sectorTo; s++) {
				printfOut("Blank check sector %d: ",s);
				if (lpcBlankCheck(fd,&fifoIn,s,s,fBank,&blank,doBlankCheck0)) {
					printfOut("%s\n",blank ? "OK" : "FAILED");
				}
				else goto failErrorClose;
			}
		}
	}

	// Command: read image if desired
	if (-1!=readN) {
		int fdImage = 1;	// default stdout
		if (fifoCanRead(&fifoO)) {
			const char *fileName = fifoReadLinear(&fifoO);
			fdImage = creat(fileName,S_IRUSR|S_IWUSR);
			if (fdImage<0) {
				printfError("ERROR: cannot create file \"%s\"\n",fileName);
				goto failErrorClose;
			}
		}

		char imageBuffer[512*1024];
		Fifo fifoImage = { imageBuffer, sizeof imageBuffer };

		printfNormal("Read FLASH [%d] ",readN);
		if (lpcRead(fd,&fifoIn,member.ispProtocol,&fifoImage,address,readN)) {
			if (ioWriteFifo(fdImage,&fifoImage)) {	// write to stdout, fd 1
				printfNormal(" OK.\n");
			}
			else {
				printfError("ERROR: Broken pipe.\n");
				goto failErrorClose;
			}
		}
		else {
			printfError("ERROR: cannot read FLASH.\n");
			goto failErrorClose;
		}
		if (fdImage!=1) close(fdImage);
	}

	// explicit erasure of sectors
	const bool eraseAll = eraseSymbol==0;
	if (eraseAll) {
		erase.a = 0;
		erase.b = lpcAddressToSector(&member, member.familyMemoryMap->addressFlash+member.sizeFlashK*1024 -1);
		if (erase.b==-1) {
			printfError("Failed to calculate last sector\n");
			goto failErrorClose;
		}
	}

	if (erase.a<=erase.b) {
		if (true
		&& lpcUnlock(fd,&fifoIn)
		&& lpcPrepareForWrite(fd,&fifoIn,erase.a,erase.b,fBank)
		&& lpcErase(fd,&fifoIn,erase.a,erase.b,fBank)) {
			fprintf(stderr,"Erased sectors %d to %d\n",erase.a,erase.b);
		}
		else {
			fprintf(stderr,"Failed to erase sectors %d to %d\n",erase.a,erase.b);
			fflush(stderr);
			goto failErrorClose;
		}
	}

	// writing
	Uint8 hexImageBuffer[512*1024];
	HexImage hexImage = { hexImageBuffer, sizeof hexImageBuffer, };
	hexImageInit(&hexImage);
	if (writeAll || writeN!=-1) {
		if (writeAll && writeN!=-1) {
			printfError("Both write N and write all requested.\n");
			goto failErrorClose;
		}

		if (!hexFileLoad(inputFile,&hexImage)) {
			printfError("ERROR: could not load file \"%s\".\n",inputFile ? inputFile : "<stdin>");
			goto failErrorClose;
		}
		if (hexImageSegments(&hexImage)!=1) {
			printfError("ERROR: %d file segments in image, but only 1 supported.\n",
				hexImageSegments(&hexImage));
			goto failErrorClose;
		}
		if (address!=0) hexImage.segments[0].address = address;	// manual override

		// handle checksum in image vector table entry #5/#7
		if (lpcChecksumFix(&hexImage,checksumVector)) {
			const int n = hexImage.segments[0].size;
			printfNormal("Image: %d = 0x%06X Bytes\n", n,n);
		}
		else {
			printfError("ERROR: image too small for (required) checksum calculation.\n");
			goto failErrorClose;
		}

		// handle code locking
		if (!lpcCrp(&hexImage,addressCrp,levelCrpAllow,levelCrpForce)) {
			printfError("ERROR: CRP problem.\n");
			goto failErrorClose;
		}

		if (lpcUnlock(fd,&fifoIn) && lpcProgram(fd,&fifoIn,&member,&hexImage,&lpcProgramConfig)) ;	// fine
		else {
			printfError("ERROR: cannot program FLASH.\n");
			goto failErrorClose;
		}
	}

	// verification

	// checksum calculation
	if (checksumRangeSymbol!=-1) {
		if (checksumRange[0]!=-1) {
			printfError("ERROR: conflicting options for checksum.\n");
			goto failErrorClose;
		}
		switch(checksumRangeSymbol) {
			case 0:	// 'file' : derive from file size.
				checksumRange[1] = hexImage.segments[0].address;
				checksumRange[0] = hexImage.segments[0].size;
				break;
			case 1:	// all : derive from flash size.
				checksumRange[1] = 0;
				checksumRange[0] = member.sizeFlashK*1024;
				break;
			default:
				printfError("INTERNAL ERROR #1\n");
				goto failErrorClose;
		}
	}
	if (checksumRange[0]>0) {
		Uint32 crc;
		if (checksumRange[1]<remapBytes		// we hit the re-mapped are
		&& !doCrcLower) {			// we do a fake test in re-mapped area only
			if (checksumRange[0]+checksumRange[1]<=remapBytes) {
				printfError("ERROR: verifying re-mapped bytes, only.\n");
				goto failErrorClose;
			}
			const Uint32 overlap = remapBytes - checksumRange[1];
			checksumRange[1] += overlap;
			checksumRange[0] -= overlap;
		}

		if (lpcReadCrc(fd,&fifoIn,checksumRange[1],checksumRange[0],&crc)) {
			printfOut("0x%08X\n",crc);
		}
		else goto failErrorClose;
	}

	// set FLASH bank
	if (setFlashBank!=-1) {
		if (lpcSetFlashBank(fd,&fifoIn,setFlashBank)) {
			printfNormal("Active flash bank set to %d.\n",setFlashBank);
		}
		else goto failErrorClose;
	}

	// program execution
	if (executeReset) {
		flowControlHook('G',"Start program by system reset, NOT ISP command GO.");
		wavePlay(fd,&waveConfiguration,WAVE_EXECUTE);
		printfNormal("Started program by system RESET\n");
	}
	else {
		if (executeSymbol!=-1) {
			switch(executeSymbol) {
				case 0:	// LMA = ARM-mode at load address
					executeAddress = address;
					break;
				case 1:	// LMA+1 = THUMB-mode at load address
					executeAddress = address+1;
					break;
				default:
					printfError("ERROR: invalid executeSymbol.\n");
					goto failErrorClose;
			}

		}
		if (executeAddress!=-1) {
			if (wavePlay(fd,&waveConfiguration,WAVE_JUMP)
			&& lpcUnlock(fd,&fifoIn)
			&& lpcGo(fd,&fifoIn,executeAddress&~1,executeAddress&1)) {
				// no return of ISP from here!
				printfNormal("Started program at address 0x%08X\n",executeAddress);
				goto returnClose;
			}
			else {
				printfError("ERROR: failed to execute.\n");
				goto failErrorClose;
			}
		}
	}
	// restore settings
	//tcsetattr(fd,TCSANOW,&terminalSettingsSaved);

	returnClose:
		close(fd);
	returnEarly:	// no file descriptor open, yet
		return 0;

	failErrorClose:
		wavePlay(fd,&waveConfiguration,WAVE_ERROR);
	failEarly:	// no file descriptor open, yet
		return 1;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

int printfDebug(const char *format, ...) {
	if (verbosity>=VERBOSITY_DEBUG) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on stderr.
 */
int printfNormal(const char *format, ...) {
	if (verbosity>=VERBOSITY_NORMAL) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

 
/** Prints out progress info and messages normally expected by the user but not directly program output.
 * Subject to redirection, normally on stderr.
 */
int vprintfNormal(const char *format, va_list args) {
	if (verbosity>=VERBOSITY_NORMAL) {
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		return n;
	}
	return 0;
}

 
/** Prints out error messages normally expected by the user in case of a possible malfunction.
 * Subject to redirection, normally on stderr.
 */
int printfWarning(const char *format, ...) {
	if (verbosity>VERBOSITY_SILENT_ERROR) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

/** Prints out error messages normally expected by the user in case of a malfunction.
 * Subject to redirection, normally on stderr.
 */
int printfError(const char *format, ...) {
	if (verbosity>VERBOSITY_SILENT_ERROR) {
		va_list args;
		va_start(args,format);
		const int n = vfprintf(stderr,format,args);
		fflush(stderr);
		va_end(args);
		return n;
	}
	return 0;
}

