//HEADER
/*
  lpcMemories-sliced.c 
  Copyright 2011,2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#include <lpcMemories.h>
#include <macros.h>

//SLICE
const LpcFamilyMember* lpcFindById(const LpcFamilyMember *const *list, Uint32 id) {
	for (;*list; list++) if ((*list)->id == id) return *list;
	return 0;
}

//SLICE
static bool startsWith(const char *prefix, const char *name) {
	while (*prefix!=0) {
		if (*prefix!=*name) return false;
		prefix++; name++;
	}
	return true;
}

const LpcFamilyMember* lpcFindByName(const LpcFamilyMember *const *list, const char *name) {
	for (;*list; list++) if (startsWith(name,(*list)->name)) return *list;
	return 0;
}

// SLICE
/*
const LpcFamilyMember* lpcFindByNumber(const LpcFamilyMember *const *list, int number) {
	for (;*list; list++) if ((*list)->number == number) return *list;
	return 0;
}
*/

//SLICE
/** Calculates what sectors are affected from operations within an address range.
 * @param sectorFrom the first sector affected
 * @param sectorTo the last sector affected
 * @param addressFrom the lowest address of the range
 * @param addressTo the highest address of the range.
 * @return true, if the addresses are within the members FLASH range
 */
bool lpcAddressRangeToSectorRange(LpcFamilyMember const *member,
	int *sectorFrom, int *sectorTo, Uint32 addressFrom, Uint32 addressTo) {

	Uint32 from,to;
	int sector;
	for (sector = 0; lpcSectorRange(member,&from,&to,sector); sector++) {
		if (from<=addressFrom && addressFrom<=to) {
			*sectorFrom = sector;
			break;
		}
	}
	for ( ; lpcSectorRange(member,&from,&to,sector); sector++) {
		if (from<=addressTo && addressTo<=to) {
			*sectorTo = sector;
			return true;
		}
	}
	return false;
}

//SLICE
int lpcAddressToSector(LpcFamilyMember const *member, Uint32 address) {
	int sectorFrom, sectorTo;
	if (lpcAddressRangeToSectorRange(member,&sectorFrom,&sectorTo,address,address)) return sectorFrom;
	else return -1;
}

//SLICE
bool lpcSectorRange(LpcFamilyMember const *member, Uint32 *addressFrom, Uint32 *addressTo, int sector) {
	const LpcFamilyMemoryMap *map = member->familyMemoryMap;

	Uint32 offsetFlash = 0;
	int subSector = 0;
	int sectorType = 0;
	for (int s=0; sectorType<ELEMENTS(map->sectorArrays) && map->sectorArrays[sectorType].sizeK!=0 ; s++) {
		if (s==sector) {
			*addressFrom = map->addressFlash + offsetFlash;
			*addressTo = *addressFrom + 1024*map->sectorArrays[sectorType].sizeK -1;
			return true;
		}
		offsetFlash += 1024*map->sectorArrays[sectorType].sizeK;
 
		// next one
		if (subSector+1 < map->sectorArrays[sectorType].n) subSector ++;
		else {
			sectorType++;
			subSector = 0;
		}
	}
	return false;	// outside FLASH bounds of device
}

//SLICE
int lpcFamilySectorHighest(LpcFamilyMemoryMap const *mm) {
	int sector = 0;
	for (int ss=0; ss<ELEMENTS(mm->sectorArrays); ++ss) {
		sector += mm->sectorArrays[ss].n;
	}
	return sector -1;
}

//SLICE
int lpcFamilySectorOffset(LpcFamilyMemoryMap const *mm, int sector) {
	int offset = 0;
	int sectorCounter = 0;
	for (int ss=0; ss<ELEMENTS(mm->sectorArrays); ++ss) {
		for (int i=0; i<mm->sectorArrays[ss].n; ++i,++sectorCounter) {
			if (sector==sectorCounter) return offset;
			offset += mm->sectorArrays[ss].sizeK * 1024;
		}
	}
	return sectorCounter==sector ? offset : -1;
}

//SLICE
Uint32 lpcCrpLevelToCode(int level) {
	switch(level) {
		case 1:	return 0x12345678u;
		case 2: return 0x87654321u;
		case 3: return 0x43218765u;
		case 4: return 0x4E697370;	// as ASCII: "Nisp" reversed
		default: return 0xFFFFFFFFu;
	}
}

//SLICE
int lpcCrpCodeToLevel(Uint32 code) {
	switch(code) {
		case 0x12345678u: return 1;
		case 0x87654321u: return 2;
		case 0x43218765u: return 3;
		case 0x4E697370u: return 4;
		default: return 0;
	}
}

//SLICE
bool lpcCrpLevelAllowed(int levelAllow, int levelDesired) {
	return	levelAllow<=3 && levelDesired<=levelAllow
		|| levelAllow==4 && (levelDesired<=0 || levelDesired==4);
}

//SLICE
bool fifoPrintLpcFamilyMember(Fifo *f, LpcFamilyMember const *m) {
	const LpcFamilyMemoryMap* mm = m->familyMemoryMap; 
	bool success =
		fifoPrintString(f,m->name)
		&& fifoPrintString(f,", ID=0x") && fifoPrintHex(f,m->id,8,8)
		&& fifoPrintString(f,"\nFLASH: ") && fifoPrintUDec(f,m->sizeFlashK,1,10)
		&& fifoPrintString(f,"kiB @ 0x") && fifoPrintHex(f,mm->addressFlash,8,8)
		&& fifoPrintString(f,"\nRAM:   ")
		;
	for (int r=0; r<ELEMENTS(m->sizeRamKs) && success; r++) if ( m->sizeRamKs[r]!=0) {
		if (r!=0) success = success && fifoPrintString(f,", ");
		success = success
		&& fifoPrintUDec(f,m->sizeRamKs[r],1,10)
		&& fifoPrintString(f,"kiB @ 0x")
		&& fifoPrintHex(f,mm->addressRams[r],8,8)
		;
	}

	success = success
		&& fifoPrintLn(f)
		&& fifoPrintString(f,"ISP data protocol:")
		&& fifoPrintString(f,m->ispProtocol==ISP_PROTOCOL_UUENCODE ? "UUENCODE" : (
			m->ispProtocol==ISP_PROTOCOL_BINARY ? "BINARY" : "UNKNOWN"))
		&& fifoPrintLn(f);
	Uint32 offset = 0;
	int sector = 0;
	for (int ss=0; offset<m->sizeFlashK*1024 && ss<ELEMENTS(mm->sectorArrays); ++ss) {
		for (int s=0; s<mm->sectorArrays[ss].n && offset<m->sizeFlashK*1024; ++s) {
			success	= success
			&& fifoPrintString(f,"Sector ")
			&& fifoPrintUDec(f,sector,2,10)
			&& fifoPrintString(f,": ")
			&& fifoPrintUDec(f,mm->sectorArrays[ss].sizeK,1,10)
			&& fifoPrintString(f,"kiB @ 0x")
			&& fifoPrintHex(f,mm->addressFlash + offset,8,8)
			&& fifoPrintString(f," (")
			&& fifoPrintUDec(f,(mm->addressFlash + offset)>>10,1,10)
			&& fifoPrintString(f,"ki)\n");
			offset += mm->sectorArrays[ss].sizeK*1024;
			sector++;
		}
	}
	success = success && fifoPrintString(f,"Copy RAM to FLASH block sizes: ");
	for (int b=0; b<ELEMENTS(mm->blockSizes); ++b) {
		if (b!=0) success = success && fifoPrintString(f,", ");
		success = success && fifoPrintUDec(f,mm->blockSizes[b],1,10);
	}
	return success && fifoPrintLn(f);
}

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap8xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, },
	.sectorArrays = { { .sizeK=1, .n=16 }, { }, },
	.blockSizes = { 64, 128, 256, 512, 1024 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap11xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, },
	.sectorArrays = { { .sizeK=4, .n=16 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap11Exx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, 0x20000000, 0x20004000 },
	.sectorArrays = { { .sizeK=4, .n=32 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap12xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, },
	.sectorArrays = { { .sizeK=4, .n=32 }, { }, },
	.blockSizes = { 4, 512, 1024, 4096 }		// LPC1200 can write very small (4B) pages/blocks, 4096B max
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap13xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, },
	.sectorArrays = { { .sizeK=4, .n=8 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap13x5_6_7 = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, 0x20000000, 0x20004000 },	// RAM1/2 powered off at boot-up
	.sectorArrays = { { .sizeK=4, .n=16 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap17xx = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, 0x2007C000, 0x20080000 },
	.sectorArrays = { { .sizeK=4, .n=16 }, { .sizeK=32, .n=14 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap1788 = {
	.addressFlash = 0,
	.addressRams = { 0x10000000, 0x20000000, },
	.sectorArrays = { { .sizeK=4, .n=16 }, { .sizeK=32, .n=14 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap2103 = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=4, .n=8 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap2106 = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=16 }, { }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap211x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=16 }, {}, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap212x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=8, .n=8 }, { .sizeK=64, .n=2 }, { .sizeK=8, .n=2},  {} },
	.blockSizes = { 256, 512, 1024, 4096 }	// unchecked
};


//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap213x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, },
	.sectorArrays = { { .sizeK=4, .n=8}, { .sizeK=32, .n=14 }, { .sizeK=4, .n=5 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap214x = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, 0x7FD00000, },
	.sectorArrays = { { .sizeK=4, .n=8}, { .sizeK=32, .n=14 }, { .sizeK=4, .n=5 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMemoryMap lpcFamilyMemoryMap23xx = {
	.addressFlash = 0,
	.addressRams = { 0x40000000, 0x7FD0C000, 0x7FE00000 },
	.sectorArrays = { { .sizeK=4, .n=8 }, { .sizeK=32, .n=14 }, { .sizeK=4, .n=6 }, },
	.blockSizes = { 256, 512, 1024, 4096 }
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers8xx[] = {
	// generic ones
	&lpcFamilyMember8,
	&lpcFamilyMember810,
	&lpcFamilyMember811,
	&lpcFamilyMember812,
	
	// real ones
	&lpcFamilyMember810_M021_FN8,
	&lpcFamilyMember811_M001_FDH16,
	&lpcFamilyMember812_M101_FDH16,
	&lpcFamilyMember812_M101_FD20,
	&lpcFamilyMember812_M101_FDH20,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers11xx[] = {
	&lpcFamilyMember11,	// generic

	&lpcFamilyMember1110_FD20_ALT1,
	&lpcFamilyMember1110_FD20_ALT2,

	&lpcFamilyMember1111_M002_FDH20_ALT1,
	&lpcFamilyMember1111_M002_FDH20_ALT2,
	&lpcFamilyMember1111_M101_FHN33,
	&lpcFamilyMember1111_M102_FHN33,
	&lpcFamilyMember1111_M103_FHN33,
	&lpcFamilyMember1111_M201_FHN33,	
	&lpcFamilyMember1111_M202_FHN33,
	&lpcFamilyMember1111_M203_FHN33,

	&lpcFamilyMember1112_M101_FHN33,
	&lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT1,
	&lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT2,
	&lpcFamilyMember1112_M102_FHN33,
	&lpcFamilyMember1112_M103_FHN33,
	&lpcFamilyMember1112_M201_FHN33,
	&lpcFamilyMember1112_M202_FHN24_FHI33_FHN33,
	&lpcFamilyMember1112_M203_FHI33_FHN33,

	&lpcFamilyMember1113_M201_FHN33,
	&lpcFamilyMember1113_M202_FHN33,
	&lpcFamilyMember1113_M203_FHN33,
	&lpcFamilyMember1113_M301_FHN33_FBD48,
	&lpcFamilyMember1113_M302_FHN33_FBD48,
	&lpcFamilyMember1113_M303_FHN33_FBD48,

	&lpcFamilyMember1114_M102_FDH28_FN28_ALT1,
	&lpcFamilyMember1114_M102_FDH28_FN28_ALT2,
	&lpcFamilyMember1114_M201_FHN33,
	&lpcFamilyMember1114_M202_FHN33,
	&lpcFamilyMember1114_M203_FHN33,
	&lpcFamilyMember1114_M301_FHN33_FBD48,
	&lpcFamilyMember1114_M302_FHI33_FHN33_FBD48_FBD100,
	&lpcFamilyMember1114_M303_FHI33_FHN33_FBD48,
	&lpcFamilyMember1114_M323_FBD48,
	&lpcFamilyMember1114_M333_FHN33_FBD48,

	&lpcFamilyMember1115_M303_FBD48,

	&lpcFamilyMember11C12_M301_FBD48,
	&lpcFamilyMember11C14_M301_FBD48,
	&lpcFamilyMember11C22_M301_FBD48,
	&lpcFamilyMember11C24_M301_FBD48,

	&lpcFamilyMember11E,
	&lpcFamilyMember11E11_M101_FN33,
	&lpcFamilyMember11E12_M201_FBD48,
	&lpcFamilyMember11E13_M301_FBD48,
	&lpcFamilyMember11E14_M401_FN33_FBD48_FBD64,
	&lpcFamilyMember11E36_M501_FN33_FBD64,
	&lpcFamilyMember11E37_M401_FBD64,
	&lpcFamilyMember11E37_M501_FBD48_FBD64,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers12xx[] = {
	&lpcFamilyMember12,
	&lpcFamilyMember1224_M101,
	&lpcFamilyMember1224_M121,
	&lpcFamilyMember1225_M301,
	&lpcFamilyMember1225_M321,
	&lpcFamilyMember1226_M301,
	&lpcFamilyMember1227_M301,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers13xx[] = {
	&lpcFamilyMember1311,
	&lpcFamilyMember1311_01,
	&lpcFamilyMember1313,
	&lpcFamilyMember1313_01,
	&lpcFamilyMember1342,
	&lpcFamilyMember1343,

	&lpcFamilyMember1315,
	&lpcFamilyMember1316,
	&lpcFamilyMember1317,
	&lpcFamilyMember1345,
	&lpcFamilyMember1346,
	&lpcFamilyMember1347,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers17xx[] = {
	&lpcFamilyMember1751,
	&lpcFamilyMember1752,
	&lpcFamilyMember1754,
	&lpcFamilyMember1756,
	&lpcFamilyMember1758,
	&lpcFamilyMember1759,
	&lpcFamilyMember1764,
	&lpcFamilyMember1765,
	&lpcFamilyMember1766,
	&lpcFamilyMember1767,
	&lpcFamilyMember1768,
	&lpcFamilyMember1769,

	&lpcFamilyMember1774,
	&lpcFamilyMember1776,
	&lpcFamilyMember1777,
	&lpcFamilyMember1778,
	&lpcFamilyMember1785,
	&lpcFamilyMember1786,
	&lpcFamilyMember1787,
	&lpcFamilyMember1788,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers210x[] = {
	&lpcFamilyMember2101,
	&lpcFamilyMember2102,
	&lpcFamilyMember2103,
	&lpcFamilyMember2104,
	&lpcFamilyMember2105,
	&lpcFamilyMember2106,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers211x[] = {
	&lpcFamilyMember2114,
	&lpcFamilyMember2124,
	0
};


//SLICE
const LpcFamilyMember* const lpcFamilyMembers213x[] = {
	&lpcFamilyMember2131,
	&lpcFamilyMember2132,
	&lpcFamilyMember2134,
	&lpcFamilyMember2136,
	&lpcFamilyMember2138,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers214x[] = {
	&lpcFamilyMember2141,
	&lpcFamilyMember2142,
	&lpcFamilyMember2144,
	&lpcFamilyMember2146,
	&lpcFamilyMember2148,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembers23xx[] = {
	&lpcFamilyMember2364,
	&lpcFamilyMember2365,
	&lpcFamilyMember2366,
	&lpcFamilyMember2367,
	&lpcFamilyMember2368,
	&lpcFamilyMember2377,
	&lpcFamilyMember2378,
	&lpcFamilyMember2378IR,
	&lpcFamilyMember2387,
	&lpcFamilyMember2388,
	0
};

//SLICE
const LpcFamilyMember* const lpcFamilyMembersXxxx[] = {
	&lpcFamilyMember8,
	&lpcFamilyMember810_M021_FN8,
	&lpcFamilyMember811_M001_FDH16,
	&lpcFamilyMember812_M101_FDH16,
	&lpcFamilyMember812_M101_FD20,
	&lpcFamilyMember812_M101_FDH20,

	&lpcFamilyMember11,	// generic

	&lpcFamilyMember1110_FD20_ALT1,
	&lpcFamilyMember1110_FD20_ALT2,

	&lpcFamilyMember1111_M002_FDH20_ALT1,
	&lpcFamilyMember1111_M002_FDH20_ALT2,
	&lpcFamilyMember1111_M101_FHN33,
	&lpcFamilyMember1111_M102_FHN33,
	&lpcFamilyMember1111_M103_FHN33,
	&lpcFamilyMember1111_M201_FHN33,	
	&lpcFamilyMember1111_M202_FHN33,
	&lpcFamilyMember1111_M203_FHN33,

	&lpcFamilyMember1112_M101_FHN33,
	&lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT1,
	&lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT2,
	&lpcFamilyMember1112_M102_FHN33,
	&lpcFamilyMember1112_M103_FHN33,
	&lpcFamilyMember1112_M201_FHN33,
	&lpcFamilyMember1112_M202_FHN24_FHI33_FHN33,
	&lpcFamilyMember1112_M203_FHI33_FHN33,

	&lpcFamilyMember1113_M201_FHN33,
	&lpcFamilyMember1113_M202_FHN33,
	&lpcFamilyMember1113_M203_FHN33,
	&lpcFamilyMember1113_M301_FHN33_FBD48,
	&lpcFamilyMember1113_M302_FHN33_FBD48,
	&lpcFamilyMember1113_M303_FHN33_FBD48,

	&lpcFamilyMember1114_M102_FDH28_FN28_ALT1,
	&lpcFamilyMember1114_M102_FDH28_FN28_ALT2,
	&lpcFamilyMember1114_M201_FHN33,
	&lpcFamilyMember1114_M202_FHN33,
	&lpcFamilyMember1114_M203_FHN33,
	&lpcFamilyMember1114_M301_FHN33_FBD48,
	&lpcFamilyMember1114_M302_FHI33_FHN33_FBD48_FBD100,
	&lpcFamilyMember1114_M303_FHI33_FHN33_FBD48,
	&lpcFamilyMember1114_M323_FBD48,
	&lpcFamilyMember1114_M333_FHN33_FBD48,

	&lpcFamilyMember1115_M303_FBD48,

	&lpcFamilyMember11C12_M301_FBD48,
	&lpcFamilyMember11C14_M301_FBD48,
	&lpcFamilyMember11C22_M301_FBD48,
	&lpcFamilyMember11C24_M301_FBD48,

	&lpcFamilyMember11E,
	&lpcFamilyMember11E11_M101_FN33,
	&lpcFamilyMember11E12_M201_FBD48,
	&lpcFamilyMember11E13_M301_FBD48,
	&lpcFamilyMember11E14_M401_FN33_FBD48_FBD64,
	&lpcFamilyMember11E36_M501_FN33_FBD64,
	&lpcFamilyMember11E37_M401_FBD64,
	&lpcFamilyMember11E37_M501_FBD48_FBD64,

	&lpcFamilyMember12,
	&lpcFamilyMember1224_M101,
	&lpcFamilyMember1224_M121,
	&lpcFamilyMember1225_M301,
	&lpcFamilyMember1225_M321,
	&lpcFamilyMember1226_M301,
	&lpcFamilyMember1227_M301,

	&lpcFamilyMember1311,
	&lpcFamilyMember1311_01,
	&lpcFamilyMember1313,
	&lpcFamilyMember1313_01,
	&lpcFamilyMember1342,
	&lpcFamilyMember1343,

	&lpcFamilyMember1315,
	&lpcFamilyMember1316,
	&lpcFamilyMember1317,
	&lpcFamilyMember1345,
	&lpcFamilyMember1346,
	&lpcFamilyMember1347,


	&lpcFamilyMember17,
	&lpcFamilyMember1751,
	&lpcFamilyMember1752,
	&lpcFamilyMember1754,
	&lpcFamilyMember1756,
	&lpcFamilyMember1758,
	&lpcFamilyMember1759,
	&lpcFamilyMember1764,
	&lpcFamilyMember1765,
	&lpcFamilyMember1766,
	&lpcFamilyMember1767,
	&lpcFamilyMember1768,
	&lpcFamilyMember1769,
	&lpcFamilyMember1774,
	&lpcFamilyMember177,
	&lpcFamilyMember1776,
	&lpcFamilyMember1777,
	&lpcFamilyMember1778,
	&lpcFamilyMember178,
	&lpcFamilyMember1785,
	&lpcFamilyMember1786,
	&lpcFamilyMember1787,
	&lpcFamilyMember1788,

	&lpcFamilyMember2101,
	&lpcFamilyMember2102,
	&lpcFamilyMember2103,
	&lpcFamilyMember2104,
	&lpcFamilyMember2105,
	&lpcFamilyMember2106,

	&lpcFamilyMember2114,
	&lpcFamilyMember2124,

	&lpcFamilyMember2131,
	&lpcFamilyMember2132,
	&lpcFamilyMember2134,
	&lpcFamilyMember2136,
	&lpcFamilyMember2138,

	&lpcFamilyMember2141,
	&lpcFamilyMember2142,
	&lpcFamilyMember2144,
	&lpcFamilyMember2146,
	&lpcFamilyMember2148,

	&lpcFamilyMember2364,
	&lpcFamilyMember2365,
	&lpcFamilyMember2366,
	&lpcFamilyMember2367,
	&lpcFamilyMember2368,
	&lpcFamilyMember2377,
	&lpcFamilyMember2378,
	&lpcFamilyMember2378IR,
	&lpcFamilyMember2387,
	&lpcFamilyMember2388,

	0	// terminator
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// LPC800 family

//SLICE
// generic type
const LpcFamilyMember lpcFamilyMember8 = {
	.name = "LPC8*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
// generic type
const LpcFamilyMember lpcFamilyMember800 = {
	.name = "LPC800*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
// generic type
const LpcFamilyMember lpcFamilyMember810 = {
	.name = "LPC810*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
// generic type
const LpcFamilyMember lpcFamilyMember811 = {
	.name = "LPC811*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
// generic type
const LpcFamilyMember lpcFamilyMember812 = {
	.name = "LPC812*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};


//SLICE
const LpcFamilyMember lpcFamilyMember810_M021_FN8 = {
	.name = "LPC810 M021 FN8",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = LPC_ID_810_M021_FN8,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
const LpcFamilyMember lpcFamilyMember811_M001_FDH16 = {
	.name = "LPC811 M001 FDH16",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_811_M001_FDH16,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
const LpcFamilyMember lpcFamilyMember812_M101_FDH16 = {
	.name = "LPC812 M101 FDH16",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_812_M101_FDH16,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
const LpcFamilyMember lpcFamilyMember812_M101_FD20 = {
	.name = "LPC812 M101 FD20",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_812_M101_FD20,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

//SLICE
const LpcFamilyMember lpcFamilyMember812_M101_FDH20 = {
	.name = "LPC812 M101 FDH20",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_812_M101_FDH20,
	.familyMemoryMap = &lpcFamilyMemoryMap8xx,
	.ispProtocol = ISP_PROTOCOL_BINARY
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// LPC1100 family


// generic one
const LpcFamilyMember lpcFamilyMember11 = {
	.name = "LPC11*",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1110_FD20_ALT1 = {
	.name = "LPC1110 FD20 (1)",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = LPC_ID_1110_FD20_ALT1,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1110_FD20_ALT2 = {
	.name = "LPC1110 FD20 (2)",
	.sizeFlashK = 4,
	.sizeRamKs = { 1, },
	.id = LPC_ID_1110_FD20_ALT2,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1111_M002_FDH20_ALT1 = {
	.name = "LPC1111 /002 FDH20 (1)",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_1111_M002_FDH20_ALT1,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1111_M002_FDH20_ALT2 = {
	.name = "LPC1111 /002 FDH20 (2)",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_1111_M002_FDH20_ALT2,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1111_M101_FHN33 = {
	.name = "LPC1111 /101 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_1111_M101_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1111_M102_FHN33 = {
	.name = "LPC1111 /102 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_1111_M102_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1111_M103_FHN33 = {
	.name = "LPC1111 /103 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_1111_M103_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1111_M201_FHN33 = {
	.name = "LPC1111 /201 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1111_M201_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};
	
const LpcFamilyMember lpcFamilyMember1111_M202_FHN33 = {
	.name = "LPC1111 /202 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1111_M202_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1111_M203_FHN33 = {
	.name = "LPC1111 /203 FHN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1111_M203_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT1 = {
	.name = "LPC1112 /102 FD20,FDH20,FDH28 (1)",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M102_FD20_FDH20_FDH28_ALT1,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT2 = {
	.name = "LPC1112 /102 FD20,FDH20,FDH28 (2)",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M102_FD20_FDH20_FDH28_ALT2,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M101_FHN33 = {
	.name = "LPC1112 /101 FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M101_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M102_FHN33 = {
	.name = "LPC1112 /102 FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M102_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M201_FHN33 = {
	.name = "LPC1112 /201 FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M201_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M202_FHN24_FHI33_FHN33 = {
	.name = "LPC1112 /202 FHN24,FHI33,FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M202_FHN24_FHI33_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M103_FHN33 = {
	.name = "LPC1112 /103 FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M103_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1112_M203_FHI33_FHN33 = {
	.name = "LPC1112 /203 FHI33,FHN33",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1112_M203_FHI33_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1113_M201_FHN33 = {
	.name = "LPC1113 /201 FHN33",
	.sizeFlashK = 24,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1113_M201_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1113_M202_FHN33 = {
	.name = "LPC1113 /202 FHN33",
	.sizeFlashK = 24,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1113_M202_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1113_M203_FHN33 = {
	.name = "LPC1113 /203 FHN33",
	.sizeFlashK = 24,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1113_M203_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1113_M301_FHN33_FBD48 = {
	.name = "LPC1113 /301 FHN33,/FBD48",
	.sizeFlashK = 24,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1113_M301_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1113_M302_FHN33_FBD48 = {
	.name = "LPC1113 /302 FHN33,FBD48",
	.sizeFlashK = 24,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1113_M302_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1113_M303_FHN33_FBD48 = {
	.name = "LPC1113 /303 FHN33,FBD48",
	.sizeFlashK = 24,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1113_M303_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1114_M102_FDH28_FN28_ALT1 = {
	.name = "LPC1114 /102 FDH28,FN28 (1)",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1114_M102_FDH28_FN28_ALT1,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M102_FDH28_FN28_ALT2 = {
	.name = "LPC1114 /102 FDH28,FN28, (2)",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1114_M102_FDH28_FN28_ALT2,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M201_FHN33 = {
	.name = "LPC1114 /201 FHN33",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1114_M201_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M202_FHN33 = {
	.name = "LPC1114 /202 FHN33",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1114_M202_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M203_FHN33 = {
	.name = "LPC1114 /203 FHN33",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1114_M203_FHN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M301_FHN33_FBD48 = {
	.name = "LPC1114 /301 FHN33,FBD48",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1114_M301_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M302_FHI33_FHN33_FBD48_FBD100 = {
	.name = "LPC1114 /302 FHI33,FHN33,FBD48,FBD100",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1114_M302_FHI33_FHN33_FBD48_FBD100,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M303_FHI33_FHN33_FBD48 = {
	.name = "LPC1114 /303 FHI33,FHN33,FBD48",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1114_M303_FHI33_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M323_FBD48 = {
	.name = "LPC1114 /323 FBD48",
	.sizeFlashK = 48,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1114_M323_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember1114_M333_FHN33_FBD48 = {
	.name = "LPC1114 /333 FHN33,FBD48",
	.sizeFlashK = 56,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1114_M333_FHN33_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember1115_M303_FBD48 = {
	.name = "LPC1115 /303 FBD48",
	.sizeFlashK = 64,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1115_M303_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


const LpcFamilyMember lpcFamilyMember11C12_M301_FBD48 = {
	.name = "LPC11C12 /301 FBD48",
	.sizeFlashK = 16,
	.sizeRamKs = { 8, },
	.id = LPC_ID_11C12_M301_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember11C14_M301_FBD48 = {
	.name = "LPC11C14 /301 FBD48",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_11C14_M301_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember11C22_M301_FBD48 = {
	.name = "LPC11C22 /301 FBD48",
	.sizeFlashK = 16,
	.sizeRamKs = { 8, },
	.id = LPC_ID_11C22_M301_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember lpcFamilyMember11C24_M301_FBD48 = {
	.name = "LPC11C24 /301 FBD48",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_11C24_M301_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};


// generic one
const LpcFamilyMember lpcFamilyMember11E = {
	.name = "LPC11E*",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap11xx,
};

const LpcFamilyMember 	lpcFamilyMember11E11_M101_FN33 = {
	.name = "LPC11E11 /101 FN33",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_11E11_M101_FN33,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E12_M201_FBD48 = {
	.name = "LPC11E12 /201 FBD48",
	.sizeFlashK = 16,
	.sizeRamKs = { 6, },
	.id = LPC_ID_11E12_M201_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E13_M301_FBD48 = {
	.name = "LPC11E13 /301 FBD48",
	.sizeFlashK = 24,
	.sizeRamKs = { 8, },
	.id = LPC_ID_11E13_M301_FBD48,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E14_M401_FN33_FBD48_FBD64 = {
	.name = "LPC11E14 /401 FN33/FBD48/FBD64",
	.sizeFlashK = 32,
	.sizeRamKs = { 8,42, },
	.id = LPC_ID_11E14_M401_FN33_FBD48_FBD64,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E36_M501_FN33_FBD64 = {
	.name = "LPC11E36 /501 FN33/FBD64",
	.sizeFlashK = 96,
	.sizeRamKs = { 8, 2, 2, },
	.id = LPC_ID_11E36_M501_FN33_FBD64,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E37_M401_FBD64 = {
	.name = "LPC11E37 /401 FBD64",
	.sizeFlashK = 128,
	.sizeRamKs = { 8, 0, 2, },	// irregularity in RAM: for IO-handler only ??
	.id = LPC_ID_11E37_M401_FBD64,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

const LpcFamilyMember 	lpcFamilyMember11E37_M501_FBD48_FBD64 = {
	.name = "LPC11E37 /501 FBD48/FBD64",
	.sizeFlashK = 128,
	.sizeRamKs = { 8, 2, 2, },
	.id = LPC_ID_11E37_M501_FBD48_FBD64,
	.familyMemoryMap = &lpcFamilyMemoryMap11Exx,
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// LPC1200 family
const LpcFamilyMember lpcFamilyMember12 = {
	.name = "LPC12*",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1224_M101 = {
	.name = "LPC1224 M101",
	.sizeFlashK = 32,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1224_M101,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1224_M121 = {
	.name = "LPC1224 M121",
	.sizeFlashK = 48,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1224_M121,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1225_M301 = {
	.name = "LPC1225 M301",
	.sizeFlashK = 64,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1225_M301,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1225_M321 = {
	.name = "LPC1225 M321",
	.sizeFlashK = 80,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1225_M321,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1226_M301 = {
	.name = "LPC1226 M301",
	.sizeFlashK = 96,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1226_M301,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};

const LpcFamilyMember lpcFamilyMember1227_M301 = {
	.name = "LPC1227 M301",
	.sizeFlashK = 128,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1227_M301,
	.familyMemoryMap = &lpcFamilyMemoryMap12xx,
};


////////////////////////////////////////////////////////////////////////////////////////////////////
// LPC1300 family

const LpcFamilyMember lpcFamilyMember13xx = {	// least common denominator
	.name = "LPC13*",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1311 = {
	.name = "LPC1311",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1311,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1311_01 = {
	.name = "LPC1311/01",
	.sizeFlashK = 8,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1311_01,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1313 = {
	.name = "LPC1313",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1313,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1313_01 = {
	.name = "LPC1313/01",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1313_01,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1342 = {
	.name = "LPC1342",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_1342,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};

const LpcFamilyMember lpcFamilyMember1343 = {
	.name = "LPC1343",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1343,
	.familyMemoryMap = &lpcFamilyMemoryMap13xx,
};


const LpcFamilyMember lpcFamilyMember1315 = {
	.name = "LPC1315",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1315,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

const LpcFamilyMember lpcFamilyMember1316 = {
	.name = "LPC1316",
	.sizeFlashK = 48,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1316,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

const LpcFamilyMember lpcFamilyMember1317 = {
	.name = "LPC1317",
	.sizeFlashK = 64,
	.sizeRamKs = { 8, 0, 2 },
	.id = LPC_ID_1317,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

const LpcFamilyMember lpcFamilyMember1345 = {
	.name = "LPC1345",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, 2, },
	.id = LPC_ID_1345,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

const LpcFamilyMember lpcFamilyMember1346 = {
	.name = "LPC1346",
	.sizeFlashK = 48,
	.sizeRamKs = { 8, 2, },
	.id = LPC_ID_1346,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

const LpcFamilyMember lpcFamilyMember1347 = {
	.name = "LPC1347",
	.sizeFlashK = 64,
	.sizeRamKs = { 8, 2, 2 },
	.id = LPC_ID_1347,
	.familyMemoryMap = &lpcFamilyMemoryMap13x5_6_7
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// LPC1700 family

//SLICE
const LpcFamilyMember lpcFamilyMember17 = {
	.name = "LPC17*",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember175 = {
	.name = "LPC175*",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1751 = {
	.name = "LPC1751",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_1751,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1752 = {
	.name = "LPC1752",
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_1752,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1754 = {
	.name = "LPC1754",
	.sizeFlashK = 128,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1754,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1756 = {
	.name = "LPC1756",
	.sizeFlashK = 256,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1756,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1758 = {
	.name = "LPC1758",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1758,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1759 = {
	.name = "LPC1759",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1759,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};


//SLICE
const LpcFamilyMember lpcFamilyMember176 = {
	.name = "LPC176*",
	.sizeFlashK = 128,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1764,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1764 = {
	.name = "LPC1764",
	.sizeFlashK = 128,
	.sizeRamKs = { 16, 16, },
	.id = LPC_ID_1764,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1765 = {
	.name = "LPC1765",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1765,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1766 = {
	.name = "LPC1766",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1766,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1767 = {
	.name = "LPC1767",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1767,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1768 = {
	.name = "LPC1768",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1768,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember1769 = {
	.name = "LPC1769",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 16, 16 },
	.id = LPC_ID_1769,
	.familyMemoryMap = &lpcFamilyMemoryMap17xx
};

const LpcFamilyMember lpcFamilyMember177 = {
	.name = "LPC177*",
	.sizeFlashK = 128,
	.sizeRamKs = { 32, 8, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1774 = {
	.name = "LPC1774",
	.sizeFlashK = 128,
	.sizeRamKs = { 32, 8, },
	.id = LPC_ID_1774,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1776 = {
	.name = "LPC1776",
	.sizeFlashK = 256,
	.sizeRamKs = { 64, 16, },
	.id = LPC_ID_1776,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1777 = {
	.name = "LPC1777",
	.sizeFlashK = 512,
	.sizeRamKs = { 64, 32, },
	.id = LPC_ID_1777,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1778 = {
	.name = "LPC1778",
	.sizeFlashK = 512,
	.sizeRamKs = { 64, 32, },
	.id = LPC_ID_1778,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember178 = {
	.name = "LPC178*",
	.sizeFlashK = 256,
	.sizeRamKs = { 64, 16, },
	.id = 0,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1785 = {
	.name = "LPC1785",
	.sizeFlashK = 256,
	.sizeRamKs = { 64, 16, },
	.id = LPC_ID_1785,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1786 = {
	.name = "LPC1786",
	.sizeFlashK = 256,
	.sizeRamKs = { 64, 16, },
	.id = LPC_ID_1786,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1787 = {
	.name = "LPC1787",
	.sizeFlashK = 512,
	.sizeRamKs = { 64, 16, },
	.id = LPC_ID_1787,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

const LpcFamilyMember lpcFamilyMember1788 = {
	.name = "LPC1788",
	.sizeFlashK = 512,
	.sizeRamKs = { 64, 16, },
	.id = LPC_ID_1788,
	.familyMemoryMap = &lpcFamilyMemoryMap1788
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC210x family
//SLICE
const LpcFamilyMember lpcFamilyMember2101 = {
	.name = "LPC2101",
	.sizeFlashK = 8,
	.sizeRamKs = { 2, },
	.id = LPC_ID_2101,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2102 = {
	.name = "LPC2102",
	.sizeFlashK = 16,
	.sizeRamKs = { 4, },
	.id = LPC_ID_2102,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2103 = {
	.name = "LPC2103",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2103,
	.familyMemoryMap = &lpcFamilyMemoryMap2103
};

//SLICE
const LpcFamilyMember lpcFamilyMember2104 = {
	.name = "LPC2104",
	.sizeFlashK = 120,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2104,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

//SLICE
const LpcFamilyMember lpcFamilyMember2105 = {
	.name = "LPC2105",
	.sizeFlashK = 120,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2105,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

//SLICE
const LpcFamilyMember lpcFamilyMember2106 = {
	.name = "LPC2106",
	.sizeFlashK = 120,
	.sizeRamKs = { 64, },
	.id = LPC_ID_2106,
	.familyMemoryMap = &lpcFamilyMemoryMap2106
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC211x family

//SLICE
const LpcFamilyMember lpcFamilyMember2114 = {
	.name = "LPC2114",
	.sizeFlashK = 120,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2114,
	.familyMemoryMap = &lpcFamilyMemoryMap211x
};


//SLICE
const LpcFamilyMember lpcFamilyMember2124 = {
	.name = "LPC2124",
	.sizeFlashK = 248,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2124,
	.familyMemoryMap = &lpcFamilyMemoryMap212x
};



////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC213x family

//SLICE
const LpcFamilyMember lpcFamilyMember2131 = {
	.name = "LPC2131",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2131,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2132 = {
	.name = "LPC2132",
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2132,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2134 = {
	.name = "LPC2134",
	.sizeFlashK = 128,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2134,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2136 = {
	.name = "LPC2136",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2136,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2138 = {
	.name = "LPC2138",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, },
	.id = LPC_ID_2138,
	.familyMemoryMap = &lpcFamilyMemoryMap213x
};


////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC214x family

//SLICE
const LpcFamilyMember lpcFamilyMember2141 = {
	.name = "LPC2141",
	.sizeFlashK = 32,
	.sizeRamKs = { 8, },
	.id = LPC_ID_2141,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2142 = {
	.name = "LPC2142",
	.sizeFlashK = 64,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2142,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2144 = {
	.name = "LPC2144",
	.sizeFlashK = 128,
	.sizeRamKs = { 16, },
	.id = LPC_ID_2144,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2146 = {
	.name = "LPC2146",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, },
	.id = LPC_ID_2146,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};

//SLICE
const LpcFamilyMember lpcFamilyMember2148 = {
	.name = "LPC2148",
	.sizeFlashK = 512,
	.sizeRamKs = { 32, 8, },
	.id = LPC_ID_2148,
	.familyMemoryMap = &lpcFamilyMemoryMap214x
};


////////////////////////////////////////////////////////////////////////////////////////////////////
//LPC23 family

//SLICE
const LpcFamilyMember lpcFamilyMember2364 = {
	.name = "LPC2364",
	.sizeFlashK = 128,
	.sizeRamKs = { 8, 8, 16 },
	.id = LPC_ID_2364,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2365 = {
	.name = "LPC2365",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2365,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2366 = {
	.name = "LPC2366",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2366,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2367 = {
	.name = "LPC2367",
	.sizeFlashK = 256,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2367,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2368 = {
	.name = "LPC2368",
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2368,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2377 = {
	.name = "LPC2377",
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16 },
	.id = LPC_ID_2377,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2378 = {
	.name = "LPC2378",
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16, },
	.id = LPC_ID_2378,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2378IR = {
	.name = "LPC2378",
	.sizeFlashK = 504,
	.sizeRamKs = { 32, 8, 16, },
	.id = LPC_ID_2378IR,		// initial revision
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2387 = {
	.name = "LPC2387",
	.sizeFlashK = 504,
	.sizeRamKs = { 64, 16, 16, },
	.id = LPC_ID_2387,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

//SLICE
const LpcFamilyMember lpcFamilyMember2388 = {
	.name = "LPC2388",
	.sizeFlashK = 504,
	.sizeRamKs = { 64, 16, 16, },
	.id = LPC_ID_2388,
	.familyMemoryMap = &lpcFamilyMemoryMap23xx
};

