/*
  lpcMemories.h - NXP LPCxxxx microcontroller memory descriptions. 
  Copyright 2011,2013 Marc Prager
 
  This file is part of the c-any library.
  c-any is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 
  c-any is published in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License along with c-any.
  If not see <http://www.gnu.org/licenses/>
 */

#ifndef __lpcMemories_h
#define __lpcMemories_h

/** @file
 * @brief FLASH and RAM map of NXPs LPCxxxx devices, Code read protection (CRP).
 *
 * A LPC FLASH consists of multiple sectors of (typically) different sizes. However, only a few sizes are used for
 * a complete FLASH memory, so we store the segmentation as a list of pairs of (size,count).
 *
 * CRP levels 0..3 indicate increasing severity/security. Higher levels introduced to model the LPC800 NO_ISP
 * level are consideres as non-orderable levels that can be enabled or not. 
 *
 * TODO: Multi flash bank devices like LPC1800/LPC4300: n FLASH memories.
 */

#include <integers.h>
#include <stdbool.h>
#include <fifoPrint.h>

enum {
	LPC_BLOCK_SIZES	=5,	///< Copy RAM to Flash block sizes, used to be 4, 5 since LPC800
};

/** The size of individual sectors within one family is the same - what varies is the total number of sectors
 * implemented in each device. This list of different sizes/counts is terminated by a (0,0) element.
 */
typedef struct __attribute__((packed)) {
	Int8	sizeK;
	Int8	n;
} LpcSectorArray;

typedef struct __attribute__((packed,aligned(4))) {
	Uint32		addressFlash;
	Uint32		addressRams[3];
	LpcSectorArray	sectorArrays[4];
	Uint16		blockSizes[LPC_BLOCK_SIZES];
} LpcFamilyMemoryMap;

extern LpcFamilyMemoryMap const
	lpcFamilyMemoryMap8xx,
	lpcFamilyMemoryMap11xx,
	lpcFamilyMemoryMap11Exx,
	lpcFamilyMemoryMap12xx,
	lpcFamilyMemoryMap13xx,
	lpcFamilyMemoryMap13x5_6_7,
	lpcFamilyMemoryMap17xx,
	lpcFamilyMemoryMap1788,		// LPC177x and LPC178x
	lpcFamilyMemoryMap2103,
	lpcFamilyMemoryMap2106,
	lpcFamilyMemoryMap211x,
	lpcFamilyMemoryMap212x,
	lpcFamilyMemoryMap213x,
	lpcFamilyMemoryMap214x,
	lpcFamilyMemoryMap23xx;

typedef enum {
	ISP_PROTOCOL_UUENCODE,		///< UUEncode was used for binary data transmission until LPC800 appeared
	ISP_PROTOCOL_BINARY,		///< NOT-encoded binary data transmission for LPC800.
} LpcIspDataProtocol;

typedef struct __attribute__((packed,aligned(4))) {
	//Uint32			number;		// used to be numeric
	const char			*name;		///< device name, like LPC2136/01
	Uint16				sizeFlashK;
	Uint16				sizeRamKs[3];
	Uint32				id;		///< CPU ID
	const LpcFamilyMemoryMap*	familyMemoryMap;
	Int8				ispProtocol;	///< protocol used to transfer data bytes.
} LpcFamilyMember;

/* If we use IAP on the microcontroller itself, we need to be able to include into the code the member's information
 * only.
 */
extern const LpcFamilyMember
	lpcFamilyMember8,	///< generic
	lpcFamilyMember800,	///< generic
	lpcFamilyMember81,	///< generic
	lpcFamilyMember810,	///< generic
	lpcFamilyMember811,	///< generic
	lpcFamilyMember812,	///< generic

	lpcFamilyMember810_M021_FN8,
	lpcFamilyMember811_M001_FDH16,
	lpcFamilyMember812_M101_FDH16,
	lpcFamilyMember812_M101_FD20,
	lpcFamilyMember812_M101_FDH20,


	lpcFamilyMember11,	///< generic
	lpcFamilyMember11E,	///< generic

	lpcFamilyMember1110_FD20_ALT1,
	lpcFamilyMember1110_FD20_ALT2,

	lpcFamilyMember1111_M002_FDH20_ALT1,
	lpcFamilyMember1111_M002_FDH20_ALT2,
	lpcFamilyMember1111_M101_FHN33,
	lpcFamilyMember1111_M102_FHN33,
	lpcFamilyMember1111_M103_FHN33,
	lpcFamilyMember1111_M201_FHN33,	
	lpcFamilyMember1111_M202_FHN33,
	lpcFamilyMember1111_M203_FHN33,

	lpcFamilyMember1112_M101_FHN33,
	lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT1,
	lpcFamilyMember1112_M102_FD20_FDH20_FDH28_ALT2,
	lpcFamilyMember1112_M102_FHN33,
	lpcFamilyMember1112_M103_FHN33,
	lpcFamilyMember1112_M201_FHN33,
	lpcFamilyMember1112_M202_FHN24_FHI33_FHN33,
	lpcFamilyMember1112_M203_FHI33_FHN33,

	lpcFamilyMember1113_M201_FHN33,
	lpcFamilyMember1113_M202_FHN33,
	lpcFamilyMember1113_M203_FHN33,
	lpcFamilyMember1113_M301_FHN33_FBD48,
	lpcFamilyMember1113_M302_FHN33_FBD48,
	lpcFamilyMember1113_M303_FHN33_FBD48,

	lpcFamilyMember1114_M102_FDH28_FN28_ALT1,
	lpcFamilyMember1114_M102_FDH28_FN28_ALT2,
	lpcFamilyMember1114_M201_FHN33,
	lpcFamilyMember1114_M202_FHN33,
	lpcFamilyMember1114_M203_FHN33,
	lpcFamilyMember1114_M301_FHN33_FBD48,
	lpcFamilyMember1114_M302_FHI33_FHN33_FBD48_FBD100,
	lpcFamilyMember1114_M303_FHI33_FHN33_FBD48,
	lpcFamilyMember1114_M323_FBD48,
	lpcFamilyMember1114_M333_FHN33_FBD48,

	lpcFamilyMember1115_M303_FBD48,

	lpcFamilyMember11C12_M301_FBD48,
	lpcFamilyMember11C14_M301_FBD48,
	lpcFamilyMember11C22_M301_FBD48,
	lpcFamilyMember11C24_M301_FBD48,


	lpcFamilyMember11E11_M101_FN33,
	lpcFamilyMember11E12_M201_FBD48,
	lpcFamilyMember11E13_M301_FBD48,
	lpcFamilyMember11E14_M401_FN33_FBD48_FBD64,
	lpcFamilyMember11E36_M501_FN33_FBD64,
	lpcFamilyMember11E37_M401_FBD64,
	lpcFamilyMember11E37_M501_FBD48_FBD64,


	lpcFamilyMember12,
	lpcFamilyMember1224_M101,
	lpcFamilyMember1224_M121,
	lpcFamilyMember1225_M301,
	lpcFamilyMember1225_M321,
	lpcFamilyMember1226_M301,
	lpcFamilyMember1227_M301,


	lpcFamilyMember13,
	lpcFamilyMember1311,
	lpcFamilyMember1311_01,
	lpcFamilyMember1313,
	lpcFamilyMember1313_01,
	lpcFamilyMember1342,
	lpcFamilyMember1343,

	lpcFamilyMember1315,
	lpcFamilyMember1316,
	lpcFamilyMember1317,
	lpcFamilyMember1345,
	lpcFamilyMember1346,
	lpcFamilyMember1347,


	lpcFamilyMember17,	///< generic
	lpcFamilyMember1700,	///< generic

	lpcFamilyMember1751,
	lpcFamilyMember1752,
	lpcFamilyMember1754,
	lpcFamilyMember1756,
	lpcFamilyMember1758,
	lpcFamilyMember1759,

	lpcFamilyMember1764,
	lpcFamilyMember1765,
	lpcFamilyMember1766,
	lpcFamilyMember1767,
	lpcFamilyMember1768,
	lpcFamilyMember1769,

	
	lpcFamilyMember177,
	lpcFamilyMember1774,
	lpcFamilyMember1776,
	lpcFamilyMember1777,
	lpcFamilyMember1778,

	lpcFamilyMember178,
	lpcFamilyMember1785,
	lpcFamilyMember1786,
	lpcFamilyMember1787,
	lpcFamilyMember1788,


	lpcFamilyMember2101,
	lpcFamilyMember2102,
	lpcFamilyMember2103,
	lpcFamilyMember2104,
	lpcFamilyMember2105,
	lpcFamilyMember2106,


	lpcFamilyMember2114,
	lpcFamilyMember2124,


	lpcFamilyMember213x,	// generic
	lpcFamilyMember2131,
	lpcFamilyMember2132,
	lpcFamilyMember2134,
	lpcFamilyMember2136,
	lpcFamilyMember2138,


	lpcFamilyMember214x,	///< generic
	lpcFamilyMember2141,
	lpcFamilyMember2142,
	lpcFamilyMember2144,
	lpcFamilyMember2146,
	lpcFamilyMember2148,


	lpcFamilyMember236x,	///< generic
	lpcFamilyMember2364,
	lpcFamilyMember2365,
	lpcFamilyMember2366,
	lpcFamilyMember2367,
	lpcFamilyMember2368,
	lpcFamilyMember2377,
	lpcFamilyMember2378,
	lpcFamilyMember2378IR,
	lpcFamilyMember2387,
	lpcFamilyMember2388
	;

extern const LpcFamilyMember * const lpcFamilyMembers8xx[];
extern const LpcFamilyMember * const lpcFamilyMembers11xx[];
extern const LpcFamilyMember * const lpcFamilyMembers11Exx[];
extern const LpcFamilyMember * const lpcFamilyMembers12xx[];
extern const LpcFamilyMember * const lpcFamilyMembers13xx[];
extern const LpcFamilyMember * const lpcFamilyMembers17xx[];
extern const LpcFamilyMember * const lpcFamilyMembers23xx[];

extern const LpcFamilyMember * const lpcFamilyMembersXxxx[];	///< All known members.

enum {
	LPC_ID_810_M021_FN8	=0x00008100,
	LPC_ID_811_M001_FDH16	=0x00008110,
	LPC_ID_812_M101_FDH16	=0x00008120,
	LPC_ID_812_M101_FD20	=0x00008121,
	LPC_ID_812_M101_FDH20	=0x00008122,


	LPC_ID_1110_FD20_ALT1				=0x0A07102B,
	LPC_ID_1110_FD20_ALT2				=0x1A07102B,

	LPC_ID_1111_M002_FDH20_ALT1			=0x0A16D02B,
	LPC_ID_1111_M002_FDH20_ALT2			=0x1A16D02B,
	LPC_ID_1111_M101_FHN33				=0x041E502B,
	LPC_ID_1111_M102_FHN33				=0x2516D02B,
	LPC_ID_1111_M103_FHN33				=0x00010013,
	LPC_ID_1111_M201_FHN33				=0x0416502B,	
	LPC_ID_1111_M202_FHN33				=0x2516902B,
	LPC_ID_1111_M203_FHN33				=0x00010012,

	LPC_ID_1112_M101_FHN33				=0x042D502B,
	LPC_ID_1112_M102_FD20_FDH20_FDH28_ALT1		=0x0A24902B,
	LPC_ID_1112_M102_FD20_FDH20_FDH28_ALT2		=0x1A24902B,
	LPC_ID_1112_M102_FHN33				=0x2524D02B,
	LPC_ID_1112_M201_FHN33				=0x0425502B,
	LPC_ID_1112_M202_FHN24_FHI33_FHN33		=0x2524902B,
	LPC_ID_1112_M103_FHN33				=0x00020023,
	LPC_ID_1112_M203_FHI33_FHN33			=0x00020022,

	LPC_ID_1113_M201_FHN33				=0x0434502B,
	LPC_ID_1113_M202_FHN33				=0x2532902B,
	LPC_ID_1113_M301_FHN33_FBD48			=0x0434102B,
	LPC_ID_1113_M302_FHN33_FBD48			=0x2532102B,
	LPC_ID_1113_M303_FHN33_FBD48			=0x00030030,
	LPC_ID_1113_M203_FHN33				=0x00030032,

	LPC_ID_1114_M102_FDH28_FN28_ALT1		=0x0A40902B,
	LPC_ID_1114_M102_FDH28_FN28_ALT2		=0x1A40902B,
	LPC_ID_1114_M201_FHN33				=0x0444502B,
	LPC_ID_1114_M202_FHN33				=0x2540902B,
	LPC_ID_1114_M203_FHN33				=0x00040042,
	LPC_ID_1114_M301_FHN33_FBD48			=0x0444102B,
	LPC_ID_1114_M302_FHI33_FHN33_FBD48_FBD100	=0x2540102B,
	LPC_ID_1114_M303_FHI33_FHN33_FBD48		=0x00040040,
	LPC_ID_1114_M323_FBD48				=0x00040060,
	LPC_ID_1114_M333_FHN33_FBD48			=0x00040070,

	LPC_ID_1115_M303_FBD48				=0x00050080,

	LPC_ID_11C12_M301_FBD48				=0x1421102B,
	LPC_ID_11C14_M301_FBD48				=0x1440102B,
	LPC_ID_11C22_M301_FBD48				=0x1431102B,
	LPC_ID_11C24_M301_FBD48				=0x1430102B,

	LPC_ID_11E11_M101_FN33				=0x293E902B,
	LPC_ID_11E12_M201_FBD48				=0x2954502B,
	LPC_ID_11E13_M301_FBD48				=0x296A102B,
	LPC_ID_11E14_M401_FN33_FBD48_FBD64		=0x2980102B,
	LPC_ID_11E36_M501_FN33_FBD64			=0x00009C41,
	LPC_ID_11E37_M401_FBD64				=0x00007C45,
	LPC_ID_11E37_M501_FBD48_FBD64			=0x00007C41,


	LPC_ID_1224_M101		= 0x3640C02B,
	LPC_ID_1224_M121		= 0x3642C02B,
	LPC_ID_1225_M301		= 0x3650C02B,
	LPC_ID_1225_M321		= 0x3652C02B,
	LPC_ID_1226_M301		= 0x3660C02B,
	LPC_ID_1227_M301		= 0x3670C02B,


	LPC_ID_1311	= 0x2C42502B,
	LPC_ID_1311_01	= 0x1816902B,
	LPC_ID_1313	= 0x2C40102B,
	LPC_ID_1313_01	= 0x1830102B,
	LPC_ID_1342	= 0x3D01402B,
	LPC_ID_1343	= 0x3D00002B,

	LPC_ID_1315	= 0x3A010523,
	LPC_ID_1316	= 0x1A018524,
	LPC_ID_1317	= 0x1A020525,
	LPC_ID_1345	= 0x28010541,
	LPC_ID_1346	= 0x08018542,
	LPC_ID_1347	= 0x08020543,


	LPC_ID_1751	= 0x25001118,
	LPC_ID_1752	= 0x25001121,
	LPC_ID_1754	= 0x25011722,
	LPC_ID_1756	= 0x25011723,
	LPC_ID_1758	= 0x25013F37,
	LPC_ID_1759	= 0x25113737,

	LPC_ID_1764	= 0x26011922,
	LPC_ID_1765	= 0x26013733,
	LPC_ID_1766	= 0x26013F33,
	LPC_ID_1767	= 0x26012837,
	LPC_ID_1768	= 0x26013F37,
	LPC_ID_1769	= 0x26113F37,

	LPC_ID_1774	= 0x27011132,
	LPC_ID_1776	= 0x27191F43,
	LPC_ID_1777	= 0x27193747,
	LPC_ID_1778	= 0x27193F47,

	LPC_ID_1785	= 0x281D1743,
	LPC_ID_1786	= 0x281D1F43,
	LPC_ID_1787	= 0x281D3747,
	LPC_ID_1788	= 0x281D3F47,


	LPC_ID_2101	= 0,	// same as LPC2103
	LPC_ID_2102	= 0,	// same as PLC2103
	LPC_ID_2103	= 0x0004FF11,

	LPC_ID_2104	= 0xFFF0FF12,
	LPC_ID_2105	= 0xFFF0FF22,
	LPC_ID_2106	= 0xFFF0FF32,



	LPC_ID_2114	= 0x0101FF12,
	LPC_ID_2124	= 0x0101FF13,



	LPC_ID_2131	= 0x0002FF01,
	LPC_ID_2132	= 0x0002FF11,
	LPC_ID_2134	= 0x0002FF12,
	LPC_ID_2136	= 0x0002FF23,
	LPC_ID_2138	= 0x0002FF25,



	LPC_ID_2141	= 0x0402FF01,
	LPC_ID_2142	= 0x0402FF11,
	LPC_ID_2144	= 0x0402FF12,
	LPC_ID_2146	= 0x0402FF23,
	LPC_ID_2148	= 0x0402FF25,



	LPC_ID_2364	= 0x1600F902,
	LPC_ID_2365	= 0x1600E823,
	LPC_ID_2366	= 0x1600F923,
	LPC_ID_2367	= 0x1600E825,
	LPC_ID_2368	= 0x1600F925,
	LPC_ID_2377	= 0x1700E825,
	LPC_ID_2378	= 0x1700FD25,
	LPC_ID_2378IR	= 0x0703FF25,	///< initial revision devices
	LPC_ID_2387	= 0x1800F935,
	LPC_ID_2388	= 0x1800FF35,
};

const LpcFamilyMember* lpcFindById(LpcFamilyMember const * const* list, Uint32 id);
//const LpcFamilyMember* lpcFindByNumber(LpcFamilyMember const * const* list, int partNumber);

/** Searches a controller by its name.
 * @param list the list of controllers.
 * @param namePrefix the name to search. This may be shortened to search for prefixes only.
 * @return the description of the controller if found, or 0 if no match.
 */
const LpcFamilyMember* lpcFindByName(LpcFamilyMember const * const* list, const char *namePrefix);

/** Calculates what sectors are affected from operations within an address range.
 * @param member LPC family member descriptor
 * @param sectorFrom the first sector affected
 * @param sectorTo the last sector affected
 * @param addressFrom the lowest address of the range
 * @param addressTo the highest address of the range.
 * @return true, if the addresses are within the members FLASH range
 */
bool lpcAddressRangeToSectorRange(LpcFamilyMember const *member,
	int *sectorFrom, int *sectorTo, Uint32 addressFrom, Uint32 addressTo);

bool lpcSectorRange(LpcFamilyMember const *member, Uint32 *addressFrom, Uint32 *addressTo, int sector);

/** Finds the sector to a given address.
 * @param member LPC family member descriptor
 * @param address the address of a single byte.
 * @return the sector that contains the byte or -1 if no such sector exists (in that device).
 */
int lpcAddressToSector(LpcFamilyMember const *member, Uint32 address);

/** Finds the number of the highest sector of the whole family.
 */
int lpcFamilySectorHighest(LpcFamilyMemoryMap const *map);

/** Finds the offset from the beginning of the FLASH.
 * @param map the families memory map
 * @param sector any valid sector number within the map or even +1 to calculate FLASH size.
 * @return the starting offset of the given sector.
 */
int lpcFamilySectorOffset(const LpcFamilyMemoryMap* map, int sector);

/** Returns the code used to enable different levels of Code Read Protection (CRP).
 * @param level a protection level, 0..3,4. 4 is for NO_ISP-mode.
 * @return the values 0xFFFFFFFF, 0x12345678, 0x87654321, 0x43218765 and 0x4E697370 for 0..4.
 */
Uint32 lpcCrpLevelToCode(int level);

/** Translates CRP codes into CRP levels.
 * @param code either a valid CRP code for levels 1..3, 4 or any other value for level 0.
 * @return CRP level 0..3, 4 (NO_ISP). 
 */
int lpcCrpCodeToLevel(Uint32 code);

/** Checks if a desired level is within the limits.
 * Level 0 allows level 0 only.
 * Level 1 allows levels 0..1 .
 * Level 2 allows level 0..2.
 * Level 3 allows levels 0..4.
 * Level 4 allows level 0 or level 4 only. Levels 3 or 4 disable ISP. Levels 1..3 disable JTAG/SWD, but level 4
 * doesn't.
 * <p>
 * Desired levels below 0 are all allowed.
 * @param levelAllow the maximum CRP level allowed.
 * @param levelDesired the desired CRP level.
 * @return true, if the desired level is allowed by the allow level.
 */
bool lpcCrpLevelAllowed(Int32 levelAllow, Int32 levelDesired);

/** Print all information about a LPCxxxx family member.
 */
bool fifoPrintLpcFamilyMember(Fifo* fifo, LpcFamilyMember const* member);

#endif
